/*
 * mod_rxtx.java
 *
 * Created on December 24, 2006, 9:32 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package modbus;
//
import	java.io.*;

/**
 *
 * @author rcw
 */
public class mod_rxtx
{
	public final static int	CONFIG_ADDRESS = 255;
	//
	final static int		MASK8 = 0x000000ff;
	final static int		MASK16 = 0x0000ffff;
	//
	int		BUFFER_SIZE = 128;
	//
	byte	protocol = mod_def.PROT_TCP;
	byte	cur_slv_adr = (byte)1;
	//
	int		last_char;
	int		rx_length;
	int		adu_length;
	//
	byte	rxBuf[ ];	// Input packets.
	byte	xxBuf[ ];	// Temporary.
	byte	txBuf[ ];	// Output packets.
	//
	InputStream dis = null;
	OutputStream dos = null;
	//
	//-------------------------------------------------------------------------
	//
	// Here to convert a binary packet to Modbus-DF1 format.
	// Returns transmit frame length.
	//
	private int BinToTx( int length , byte src[ ] , byte dst[ ] )
	{
		int	    i , src_idx , dst_idx ;
		int	    frame_length = 0;
		byte	    dchar;

		// Zero indexes.
		src_idx = 0;
		dst_idx = 0;
		// Which protocol?
		switch ( protocol )
		{
			case mod_def.PROT_DF1:
				// Store leading DLE/STX.
				dst[ dst_idx++ ] = mod_def.MOD_DLE;
				dst[ dst_idx++ ] = mod_def.MOD_STX;
				// Transfer the rest of the packet.
				for ( i = length ; i > 0 ; --i )
				{
					// Get character.
					dchar = src[ src_idx++ ];
					// If this is a DLE.
					if ( dchar == mod_def.MOD_DLE )
					{
						// Add a DLE.
						dst[ dst_idx++ ] = dchar;
						// Increment length.
						++frame_length;
					}
					// Store character.
					dst[ dst_idx++ ] = dchar;
					// Increment length.
					++frame_length;
				}
				// Store trailing DLE/ETX.
				dst[ dst_idx++ ] = mod_def.MOD_DLE;
				dst[ dst_idx++ ] = mod_def.MOD_ETX;
				// Adjust frame length and return.
				frame_length += 4;
				break;
			case mod_def.PROT_TCP:
				// Transfer binary data frame.
				System.arraycopy(src , 0 , dst , 0 , length);
				// Set frame length.
				frame_length = length;
				break;
			default:
				// NOTREACHED.
				break;
		}
		// And return.
		return frame_length;
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to convert a Modbus DF1 packet to binary frame.
	// Returns binary frame length if successful.
	// Returns -1 if:
	//	- The received frame doesn't begin with DLE/STX.
	//	- If it encounters an unknown DLE/xxx sequence.
	//	- If the received frame doesn't end in DLE/ETX.
	//	- If there are more characters after a DLE/ETX sequence is received.
	//
	private int RxToBin( int length , byte src[ ] , byte dst[ ]  )
	{
		int		i , src_idx , dst_idx;
		int		end_of_packet;
		int		frame_length;
		byte		this_chr;
		byte            last_chr;

		// Keep Java from bitching.
		frame_length = 0;
		// Which protocol?
		switch ( protocol )
		{
			case mod_def.PROT_DF1:
				// Error if first two characters aren't DLE STX.
				if ( src[ 0 ] != mod_def.MOD_DLE )
					return -1;
				if ( src[ 1 ] != mod_def.MOD_STX )
					return -1;
				// Initialize array indexes.
				src_idx = 2;
				dst_idx = 0;
				// Adjust receive length; remove leading DLE/STX.
				length -= 2;
				// Reset end-of-packet flag.
				end_of_packet = 0;
				// Reset last char..
				last_chr = 0;
				// Transfer data and remove DLE pairs.
				i = length;
				while ( i-- > 0 )
				{
					// Get a character.
					this_chr = src[ src_idx++ ];
					// If last char was DLE.
					if ( last_chr == mod_def.MOD_DLE )
					{
						// If this one is also a DLE.
						if ( this_chr == mod_def.MOD_DLE )
						{
							// Put one DLE in dest.
							dst[ dst_idx++ ] = mod_def.MOD_DLE;
							// Decrement receive counter.
							--length;
							// Reset last char.
							last_chr = 0;
							// And continue;
							continue;
						}
						// Else if this is the end.
						else if ( this_chr == mod_def.MOD_ETX )
						{
							// Error if this isn't the last character.
							if ( i != 0 )
								return -1;
							// Set end-of-packet flag.
							end_of_packet = 1;
							// And continue.
							continue;
						}
						// Else unknown sequence.
						else
							return -1;
					}
					// If this character is a DLE
					if ( this_chr == mod_def.MOD_DLE )
					{
						// Make it the last character received.
						last_chr = mod_def.MOD_DLE;
						// And continue with loop.
						continue;
					}
					// Transfer character.
					dst[ dst_idx++ ] = this_chr;
					// Make last char.
					last_chr = this_chr;
				}
				// Error if end-of-packet not found.
				if ( end_of_packet == 0 )
					return -1;
				// Return with real packet length; adjust for trailing DLE ETX.
				frame_length = length - 2;
				break;
			case mod_def.PROT_TCP:
				// Set frame length.
				frame_length = length;
				// Transfer binary data frame.
				System.arraycopy(src , 0 , dst , 0 , length);
				break;
			default:
				// NOTREACHED.
				break;
		}
		// And return.
		return frame_length;

	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to reset in preparation for receiving a new packet.
	//
	private void ResetRx()
	{
		// Reset variables.
		last_char = 0;
		rx_length = 0;
		adu_length = 0;
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to add a character to the receive buffer.
	// Returns packet length if packet complete, '0' otherwise.
	//
	private int RxChar( int dchar )
	{
		// Store it.
        if ( rx_length < BUFFER_SIZE )
        {
            rxBuf[ rx_length++ ] = (byte)dchar;
        }
        // Else overflow; abort.
        else
        {
            ResetRx();
            return -1;
        }
		// Which protocol?
		switch ( protocol )
		{
			case mod_def.PROT_DF1:
				//
				// NOTE: Added this code to make sure we start a
				// packet with DLE/STX. This is a kluge.
				//
				// How many characters?
				switch ( rx_length )
				{
					case 1:
						// If first character is not a DLE.
						if ( dchar != mod_def.MOD_DLE )
						{
							// Reset the count.
							rx_length = 0;
							// Null the character.
							dchar = 0;
						}
						break;
					case 2:
						// If second character is not a STX.
						if ( dchar != mod_def.MOD_STX )
						{
							// Reset the count.
							rx_length = 0;
							// Null the character.
							dchar = 0;
						}
						break;
					default:
						// If last character was a DLE.
						if ( last_char == mod_def.MOD_DLE )
						{
							// And if the this character is also a DLE.
							if ( dchar == mod_def.MOD_DLE )
							{
								// Null this character, as we've received
								// two DLE's in a row. If we keep this as
								// last character, we could prematurely
								// end a packet if the next received char
								// is a '3'.
								dchar = 0;
							}
							// Else if current character is an ETX.
							else if ( dchar == mod_def.MOD_ETX )
							{
								// Packet is ready.
								return rx_length;
							}
						}
					break;
				}
				break;
			case mod_def.PROT_TCP:
				// If we haven't gotten the adu_length yet.
				if ( adu_length == 0 )
				{
					// If the length has been received.
					if ( rx_length == 6 )
					{
						// Get the length field.
						adu_length = ( rxBuf[4] & 0x000000ff ) << 8;
						adu_length += rxBuf[5] & 0x000000ff;
						adu_length += 6;
					}
				}
				// Else if we've gotten all the characters.
				else if ( rx_length >= adu_length )
				{
					// Packet is ready.
					return rx_length;
				}
				break;
			default:
				// Should never get here.
				break;
		}
		// Store last character.
		last_char = dchar;
		// And return.
		return 0;
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to wait for a packet.
	//
	private int WaitPacket()
	{
		int	    dchar;

		// Setup for packet receive operation.
		ResetRx();
		// Wait for end of packet.
		while ( true )
		{
			// Get character, return if socket closed.
			try
			{
				dchar = dis.read();
			}
			catch (IOException e)
			{
				// If we get here, the input stream has probably
				// closed, so we will exit.
				return -1;
			}
			// Add character to receive buffer.
            dchar = RxChar( dchar );
            // If overflow.
            if ( dchar < 0 )
            {
                ResetRx();
                return -1;
            }
            // If packet received.
			if ( dchar != 0 )
			{
				break;
			}
		}
		// And return.
		return rx_length;
	}
	//
	//-----------------------------------------------------------------------------
	//
	// Here to convert a null terminated byte array to a String.
	// Returns the index of the start of the next string within the byte array.
	//
	private int BytesToString( byte[ ] src , int b_idx , mod_ex mex , int s_idx )
	{
		byte	dchar;

		// Start with null string.
		mex.s[ s_idx ] = "";
		// Loop until null terminator found.
		while ( true )
		{
			// Get character, break if null.
			if ( ( dchar = src[ b_idx++ ] ) == 0 )
			{
				break;
			}
			// Add character to string.
			mex.s[ s_idx ] += Character.toString( (char)dchar );
		}
		return b_idx;
	}
	//
	//-----------------------------------------------------------------------------
	//
	// Here to convert a string to a null terminated byte array.
	// Returns with the index within dst[ ] of the next available byte for
	// the next string to be added.
	//
	private int StringToBytes( byte[ ] dst , int b_idx , mod_ex mex , int s_idx )
	{
		int	    i , length;

		length = mex.s[ s_idx ].length();
		for ( i = 0 ; i < length ; ++i )
		{
			dst[ b_idx++ ] = (byte)mex.s[ s_idx ].charAt( i );
		}
		dst[ b_idx++ ] = 0;
		return b_idx;
	}
	//
	//-----------------------------------------------------------------------------
	//
	// Master routine to read/write data items.
	//
	// NOTE: 'itm_adr' is 1-based value; i.e. register 1 is passed as 1.
	//       This routine converts the register to 0-based.
	//
	// Returns 0 if success.
	// Returns -1 and stores Modbus exception code in data.b[0] if problem
	// with packet or transaction.
	//
	public int MasterTx( mod_ex mex )
	{
	    int    tmpval;
	    int    i , frame_length;
	    int    rx_idx , xx_idx;
	    byte   byte_val , byte_msk , byte_cnt;

		// _DBUG: Testing for rx bytes in stream before starting a new transfer.
		tmpval = 0;
		try
		{
			tmpval = dis.available();
		}
		catch (IOException e)
		{
		}
		if (tmpval != 0)
		{
			try
			{
				dis.read(xxBuf);
			}
			catch (IOException e)
			{
			}
		}

		// Initialize indexes.
		rx_idx = xx_idx = 0;
		// If this is Modbus/TCP.
		if ( protocol == mod_def.PROT_TCP )
		{
			// Transaction id.
			xxBuf[ xx_idx++ ] = (byte)( mex.tran_id >> 8 );
			xxBuf[ xx_idx++ ] = (byte)mex.tran_id;
			// Protocol id.
			xxBuf[ xx_idx++ ] = (byte)( mex.prot_id >> 8 );
			xxBuf[ xx_idx++ ] = (byte)mex.prot_id;
			// Move past mbap_len.
			xx_idx += 2;
		}
		// Slave address.
		xxBuf[ xx_idx++ ] = mex.slv_adr;
		// Function.
		xxBuf[ xx_idx++ ] = mex.fcn;
		// Convert item address to 0-based, and store it big-endian.
		tmpval = mex.itm_adr - 1;
		xxBuf[ xx_idx++ ] = (byte)(tmpval >> 8);
		xxBuf[ xx_idx++ ] = (byte)tmpval;
		// Convert and store rest of data.
		switch ( (int)mex.fcn )
		{
			case mod_def.READ_COILS:
			case mod_def.READ_DISCRETE_INPUTS:
			case mod_def.READ_HOLDING_REGISTERS:
			case mod_def.READ_INPUT_REGISTERS:
				//
				// Store item quantity.
				xxBuf[ xx_idx++ ] = (byte)( mex.itm_qty >> 8 );
				xxBuf[ xx_idx++ ] = (byte)mex.itm_qty;
				//
				break;
				//
			case mod_def.WRITE_SINGLE_COIL:
			    //
			    // Store data. '1' is 0xff00.
			    if ( mex.b[0] != 0 )
			        tmpval = 0x0ff00;
			    else
			        tmpval = 0;
                xxBuf[ xx_idx++ ] = (byte)( tmpval >> 8 );
                xxBuf[ xx_idx++ ] = (byte)tmpval;
                //
                break;
                //
			case mod_def.WRITE_SINGLE_REGISTER:
				//
				// Store item data.
				xxBuf[ xx_idx++ ] = (byte)( mex.w[ 0 ] >> 8 );
				xxBuf[ xx_idx++ ] = (byte)mex.w[ 0 ];
				//
				break;
				//
			case mod_def.WRITE_MULTIPLE_COILS:
				//
				// Store item quantity.
				xxBuf[ xx_idx++ ] = (byte)( mex.itm_qty >> 8 );
				xxBuf[ xx_idx++ ] = (byte)mex.itm_qty;
				// Store number of bytes.
				byte_cnt = (byte)( mex.itm_qty / 8 );
				if ( ( mex.itm_qty & 0x07 ) != 0 )
					++byte_cnt;
				xxBuf[ xx_idx++ ] = (byte)byte_cnt;
				// Store data.
				byte_val = 0;
				byte_msk = 1;
				for ( i = 0 ; i < mex.itm_qty ; ++i )
				{
				    if ( mex.b[i] != 0)
				        byte_val |= byte_msk;
				    if ( byte_msk == 0x80 )
				    {
				        xxBuf[xx_idx++] = byte_val;
				        byte_val = 0;
				        byte_msk = 1;
				        --byte_cnt;
				    }
				    else
				        byte_msk <<= 1;
				}
                if ( byte_cnt != 0 )
                {
                    xxBuf[xx_idx++] = byte_val;
                }
				//
				break;
				//
			case mod_def.WRITE_MULTIPLE_REGISTERS:
				//
				// Store item quantity.
				xxBuf[ xx_idx++ ] = (byte)( mex.itm_qty >> 8 );
				xxBuf[ xx_idx++ ] = (byte)mex.itm_qty;
				// Store number of bytes.
				xxBuf[ xx_idx++ ] = (byte)( mex.itm_qty * 2 );
				// Store data.
				for ( i = 0 ; i < mex.itm_qty ; ++i )
				{
					xxBuf[ xx_idx++ ] = (byte)( mex.w[ i ] >> 8 );
					xxBuf[ xx_idx++ ] = (byte)mex.w[ i ];
				}
				//
				break;
				//
				//------------------------------------------------------------------
				//  Modbus RPC section, MASTER TRANSMIT.
				//------------------------------------------------------------------
				//
			case mod_def.RD_REGS_STR:
				//
				//                  \/
				// Format --> S F AA QQ
				// Format <-- S F AA QQ B..B\0B..B\0...
				//
				// Store item quantity.
				xxBuf[ xx_idx++ ] = (byte)( mex.itm_qty >> 8 );
				xxBuf[ xx_idx++ ] = (byte)mex.itm_qty;
				//
				break;
				//
			case mod_def.WR_REGS_STR:
				//
				// Sending a series of null terminated strings.
				// mex.itm_qty should be number of strings, not length of packet.
				// Strings to send are in mex.s[ ].
				//
				//                  \/
				// Format --> S F AA QQ B..B\0B..B\0...
				// Format <-- S F AA
				//
				// Store item quantity.
				xxBuf[ xx_idx++ ] = (byte)( mex.itm_qty >> 8 );
				xxBuf[ xx_idx++ ] = (byte)mex.itm_qty;
				// Transfer string to byte array.
				for ( i = 0 ; i < mex.itm_qty ; ++i )
				{
					xx_idx = StringToBytes( xxBuf , xx_idx , mex , i );
				}
				//
				break;
				//
			case mod_def.MRPC_FCN:
				//
				//                  \/
				// Format --> S F AA QQ LLL...
				// Format <-- S F AA QQ LLL...
				//
				// Store item quantity.
				xxBuf[ xx_idx++ ] = (byte)( mex.itm_qty >> 8 );
				xxBuf[ xx_idx++ ] = (byte)mex.itm_qty;
				// Store data.
				for ( i = 0 ; i < mex.itm_qty ; ++i )
				{
					xxBuf[ xx_idx++ ] = (byte)( mex.l[ i ] >> 24 );
					xxBuf[ xx_idx++ ] = (byte)( mex.l[ i ] >> 16 );
					xxBuf[ xx_idx++ ] = (byte)( mex.l[ i ] >> 8 );
					xxBuf[ xx_idx++ ] = (byte)mex.l[ i ];
				}
				//
				break;
				//
			case mod_def.RD_REGS_LIST:
				//
				//                  \/
				// Format --> S F AA QQ RR RR...
				// Format <-- S F AA QQ B..B\0B..B\0...
				//
				// Store item quantity.
				xxBuf[ xx_idx++ ] = (byte)( mex.itm_qty >> 8 );
				xxBuf[ xx_idx++ ] = (byte)mex.itm_qty;
				// Transfer register list.
				for ( i = 0 ; i < mex.itm_qty ; ++i )
				{
					xxBuf[ xx_idx++ ] = (byte)( mex.w[ i ] >> 8 );
					xxBuf[ xx_idx++ ] = (byte)mex.w[ i ];
				}
				//
				break;
				//
			case mod_def.RD_OBJ:
			    //
			    //             \/
                // Req : S F QQ IIII PP PP...
                // Rsp : S F C c..c\0 c..c\0...         (Success)
                // Rsp : S F R                          (Failure)
			    //
			case mod_def.WR_OBJ:
			    //
			    //             \/
                // Req : S F QQ IIII PP c..c\0 PP c..c\0...
                // Rsp : S F C                          (Success)
                // Rsp : S F R                          (Failure)
			    //
			    // NOTE: mex data storage:
			    //   mex.l[0] = object instance
			    //   mex.w[n] = property numbers
			    //   mex.s[n] = property strings
			    //
			    // Note that mex.itm_adr actually has the quantity and
			    // index flag.
			    // Get the actual number of properties to read/write.
			    mex.itm_qty = (short)( mex.itm_adr & 0x7fff );
			    // Object instance in network byte order.
			    xxBuf[ xx_idx++ ] = (byte)(mex.l[0] >> 24);
                xxBuf[ xx_idx++ ] = (byte)(mex.l[0] >> 16);
                xxBuf[ xx_idx++ ] = (byte)(mex.l[0] >> 8);
                xxBuf[ xx_idx++ ] = (byte)mex.l[0];
                // For each property.
                for ( i = 0 ; i < mex.itm_qty ; ++i )
                {
                    // Property number in network byte order.
                    xxBuf[ xx_idx++ ] = (byte)(mex.w[i] >> 8);
                    xxBuf[ xx_idx++ ] = (byte)mex.w[i];
                    // If writing.
                    if ( (int)mex.fcn == mod_def.WR_OBJ )
                    {
                        // Property string.
                        xx_idx = StringToBytes( xxBuf , xx_idx , mex , i );
                    }
                }
			    break;
				//
			default:
				//
				// Mark as error.
				mex.fcn |= 0x80;
			mex.b[0] = mod_def.ILLEGAL_FUNCTION;
			// And return.
			return -1;
		}
		// If this is Modbus/TCP.
		if ( protocol == mod_def.PROT_TCP )
		{
			// Get length for MBAP header and store it.
			frame_length = xx_idx - 6;
			xxBuf[ 4 ] = (byte)(frame_length >> 8 );
			xxBuf[ 5 ] = (byte)frame_length;
		}
		// Create frame to transmit.
		frame_length = BinToTx( xx_idx , xxBuf , txBuf );
		// Send frame; return if error.
		try
		{
			dos.write( txBuf , 0 , frame_length );
		}
		catch (IOException e)
		{
			//
			// If we get here, the input stream has probably
			// closed, so we will exit.
			//
			// Store error string if register list operation.
			if ( mex.fcn == mod_def.RD_REGS_LIST )
				mex.s[ 0 ] = mod_def.ErrStr[ mod_def.DATA_TX_ERROR ];
			else
				mex.b[0] = mod_def.DATA_TX_ERROR;
			// Mark as error.
			mex.fcn |= 0x80;
			// And return.
			return -1;
		}
		//
		// If this was a broadcast, we do not wait for a reply!!!
		//
		if ( mex.slv_adr == 0 )
			return 0;
		// Wait for reply.
		frame_length = WaitPacket();
		// Return if error.
		if ( frame_length == -1 )
		{
			// Store error string if register list operation.
			if ( mex.fcn == mod_def.RD_REGS_LIST )
				mex.s[ 0 ] = mod_def.ErrStr[ mod_def.DATA_RX_ERROR ];
			else
				mex.b[0] = mod_def.DATA_RX_ERROR;
			// Mark as error.
			mex.fcn |= 0x80;
			return -1;
		}
		// Convert to binary.
		frame_length = RxToBin( frame_length , rxBuf , rxBuf );
		// Zero buffer index.
		rx_idx = 0;
		// If this is Modbus/TCP.
		if ( protocol == mod_def.PROT_TCP )
		{
			// Get transaction id.
			mex.tran_id = (short)( rxBuf[ rx_idx++ ] << 8 );
			mex.tran_id |= rxBuf[ rx_idx++ ] & 0x00ff;
			// Get protocol id.
			mex.prot_id = (short)( rxBuf[ rx_idx++ ] << 8 );
			mex.prot_id |= rxBuf[ rx_idx++ ] & 0x00ff;
			// Get mbap length field.
			mex.mbap_len = (short)( rxBuf[ rx_idx++ ] << 8 );
			mex.mbap_len |= rxBuf[ rx_idx++ ] & 0x00ff;
		}
		// Store slave address.
		mex.slv_adr = rxBuf[ rx_idx++ ];
		// Store function (with possible error marker).
		mex.fcn = rxBuf[ rx_idx++ ];
		// If error response.
		if ( ( mex.fcn & 0x80 ) != 0 )
		{
			// Store error code if not register list operation.
			// Register list op will have error string in s[ ].
			if ( ( mex.fcn & 0x7f ) != mod_def.RD_REGS_LIST )
			{
				// Store error code.
				mex.b[0] = rxBuf[ rx_idx ];
			}
			// Return with error.
			return -1;
		}
		// Process return data.
		switch ( (int)mex.fcn )
		{
			case mod_def.WRITE_SINGLE_COIL:
			case mod_def.WRITE_SINGLE_REGISTER:
			case mod_def.WRITE_MULTIPLE_COILS:
			case mod_def.WRITE_MULTIPLE_REGISTERS:
				//
				// Break if writing data.
				break;
				//
			case mod_def.READ_COILS:
			case mod_def.READ_DISCRETE_INPUTS:
				//
			    // Skip byte count.
			    ++rx_idx;
			    // Set bitmask.
			    byte_msk = 1;
			    // For each item.
			    for ( i = 0 ; i < mex.itm_qty ; ++i )
			    {
			        if ( (rxBuf[rx_idx] & byte_msk) != 0 )
			            mex.b[i] = 1;
			        else
			            mex.b[i] = 0;
			        if ( byte_msk == 0x80 )
			        {
			            ++rx_idx;
			            byte_msk = 1;
			        }
			        else
			            byte_msk <<= 1;
			    }
				//
				break;
				//
			case mod_def.READ_HOLDING_REGISTERS:
			case mod_def.READ_INPUT_REGISTERS:
				//
				// Skip byte count.
				++rx_idx;
				// Get register values.
				for ( i = 0 ; i < mex.itm_qty ; ++i )
				{
					mex.w[ i ]  = (short)( rxBuf[ rx_idx++ ] << 8 );
					mex.w[ i ] |= rxBuf[ rx_idx++ ] & 0x00ff;
				}
				break;
				//
				//------------------------------------------------------------------
				//  Modbus RPC section, MASTER RECEIVE.
				//------------------------------------------------------------------
				//
			case mod_def.RD_REGS_STR:
			case mod_def.RD_REGS_LIST:
				//
				// Receiving a series of null terminated strings.
				//
				// Format --> S F AA QQ
				//               \/
				// Format <-- S F AA QQ B..B\0B..B\0...
				//
				// Skip address field.
				rx_idx += 2;
				// Store item quantity.
				mex.itm_qty = (short)( rxBuf[ rx_idx++ ] << 8 );
				mex.itm_qty |= rxBuf[ rx_idx++ ] & 0x00ff;
				// Transfer bytes to strings.
				for ( i = 0 ; i < mex.itm_qty ; ++i )
				{
					rx_idx = BytesToString( rxBuf , rx_idx , mex , i );
				}
				//
				break;
				//
			case mod_def.WR_REGS_STR:
				//
				// No action required.
				//
				break;
				//
			case mod_def.MRPC_FCN:
				//
				// Format --> S F AA QQ LLL...
				//               \/
				// Format <-- S F AA QQ LLL...
				//
				// Skip address field.
				rx_idx += 2;
				// Store number of returned values.
				mex.itm_qty = (short)( rxBuf[ rx_idx++ ] << 8 );
				mex.itm_qty |= rxBuf[ rx_idx++ ] & 0x00ff;
				// Transfer returned values.
				for ( i = 0 ; i < mex.itm_qty ; ++i )
				{
					mex.l[ i ]  = ( rxBuf[ rx_idx++ ] << 24 ) & 0xff000000;
					mex.l[ i ] |= ( rxBuf[ rx_idx++ ] << 16 ) & 0x00ff0000;
					mex.l[ i ] |= ( rxBuf[ rx_idx++ ] << 8  ) & 0x0000ff00;
					mex.l[ i ] |= rxBuf[ rx_idx++ ]           & 0x000000ff;
				}
				//
				break;
                //
            case mod_def.RD_OBJ:
                //
                // Req : S F QQ IIII PP PP...
                //          \/
                // Rsp : S F C c..c\0 c..c\0...         (Success)
                //
            case mod_def.WR_OBJ:
                //
                // Req : S F QQ IIII PP c..c\0 PP c..c\0...
                //          \/
                // Rsp : S F C                          (Success)
                //
                // NOTE: mex data storage:
                //   mex.l[0] = object instance
                //   mex.w[n] = property numbers
                //   mex.s[n] = property strings
                //
                // Nothing to do if writing.
                if ( (int)mex.fcn == mod_def.WR_OBJ )
                    break;
                // Get number of strings.
                mex.itm_qty = (short)rxBuf[ rx_idx++ ];
                // For each string.
                for ( i = 0 ; i < mex.itm_qty ; ++i )
                {
                    // Transfer the string.
                    rx_idx = BytesToString( rxBuf , rx_idx , mex , i );
                }
				//
                break;
                //
			default:
				break;
		}
		// And return.
		return 0;
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a 16 bit value.
	// Returns 0 if success, -1 if failure.
	//
	public int ReadReg16( mod_ex mex , int chn , int reg )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.READ_HOLDING_REGISTERS;
		mex.itm_adr = (short)mex.RegIdx( chn , reg );
		mex.itm_qty = 1;
		MasterTx( mex );
		return mex.w[0];
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a 16 bit value.
	// Returns 0 if success, -1 if failure.
	//
	public int ReadReg16( mod_ex mex , int reg )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.READ_HOLDING_REGISTERS;
		mex.itm_adr = (short)reg;
		mex.itm_qty = 1;
		MasterTx( mex );
		return mex.w[0];
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write a 16 bit value.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteReg16( mod_ex mex , int chn , int reg , int value )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.WRITE_MULTIPLE_REGISTERS;
		mex.itm_adr = (short)mex.RegIdx( chn , reg );
		mex.itm_qty = 1;
		mex.w[0] = (short)value;
		return MasterTx( mex );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write a 16 bit value.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteReg16( mod_ex mex , int reg , int value )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.WRITE_MULTIPLE_REGISTERS;
		mex.itm_adr = (short)reg;
		mex.itm_qty = 1;
		mex.w[0] = (short)value;
		return MasterTx( mex );
	}
	//
	//-------------------------------------------------------------------------
	// String operations; with and w/o channel parameter.
	//-------------------------------------------------------------------------
	//
	// Here to read a single string value.
	// Result is in mex.s[0] if success.
	// Returns 0 if success, -1 if failure.
	//
	public int ReadString( mod_ex mex , int chn , int reg )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.RD_REGS_STR;
		mex.itm_adr = (short)mex.RegIdx( chn , reg );
		mex.itm_qty = 1;
		return MasterTx( mex );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a single string value.
	// Result is in mex.s[0] if success.
	// Returns 0 if success, -1 if failure.
	//
	public int ReadString( mod_ex mex , int reg )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.RD_REGS_STR;
		mex.itm_adr = (short)reg;
		mex.itm_qty = 1;
		return MasterTx( mex );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write a string value.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteString( mod_ex mex , int chn , int reg , String val )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.WR_REGS_STR;
		mex.itm_adr = (short)mex.RegIdx( chn , reg );
		mex.itm_qty = 1;
		mex.s[0] = val;
		return MasterTx( mex );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write a string value.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteString( mod_ex mex , int reg , String val )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.WR_REGS_STR;
		mex.itm_adr = (short)reg;
		mex.itm_qty = 1;
		mex.s[0] = val;
		return MasterTx( mex );
	}
	//
	//-------------------------------------------------------------------------
	// Boolean operations; with and w/o channel parameter.
	//-------------------------------------------------------------------------
	//
	// Here to read a register as an integer value.
	// Value returned in mex.l[0];
	// Returns 0 if success, -1 if failure.
	//
	public boolean ReadBool( mod_ex mex , int chn , int reg )
	{
		ReadString( mex , chn , reg );
		if ( mex.s[0].equals( "0") )
			return false;
		else
			return true;
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a register as an integer value.
	// Value returned in mex.l[0];
	// Returns 0 if success, -1 if failure.
	//
	public boolean ReadBool( mod_ex mex , int reg )
	{
		ReadString( mex , reg );
		if ( mex.s[0].equals( "0") )
			return false;
		else
			return true;
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write an integer value to a register.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteBool( mod_ex mex , int chn , int reg , boolean val )
	{
		if ( val == true )
			return WriteString( mex , chn , reg , "1" );
		else
			return WriteString( mex , chn , reg , "0" );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write an integer value to a register.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteBool( mod_ex mex , int reg , boolean val )
	{
		if ( val == true )
			return WriteString( mex , reg , "1" );
		else
			return WriteString( mex , reg , "0" );
	}
	//
	//-------------------------------------------------------------------------
	// Integer operations; with and w/o channel parameter.
	//-------------------------------------------------------------------------
	//
	// Here to read a register as an integer value.
	// Value returned in mex.l[0];
	// Returns 0 if success, -1 if failure.
	//
	public int ReadInt( mod_ex mex , int chn , int reg )
	{
		ReadString( mex , chn , reg );
		return Integer.parseInt( mex.s[0] );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a register as an integer value.
	// Value returned in mex.l[0];
	// Returns 0 if success, -1 if failure.
	//
	public int ReadInt( mod_ex mex , int reg )
	{
		ReadString( mex , reg );
		return Integer.parseInt( mex.s[0] );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write an integer value to a register.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteInt( mod_ex mex , int chn , int reg , int val )
	{
		return WriteString( mex , chn , reg , Integer.toString( val ) );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write an integer value to a register.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteInt( mod_ex mex , int reg , int val )
	{
		return WriteString( mex , reg , Integer.toString( val ) );
	}
	//
	//-------------------------------------------------------------------------
	// Floating point operations; with and w/o channel parameter.
	//-------------------------------------------------------------------------
	//
	// Here to read a register as a floating point value.
	// Value returned in mex.f[0];
	// Returns 0 if success, -1 if failure.
	//
	public float ReadFloat( mod_ex mex , int chn , int reg )
	{
		ReadString( mex , chn , reg );
		return Float.parseFloat( mex.s[0] );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a register as a floating point value.
	// Value returned in mex.f[0];
	// Returns 0 if success, -1 if failure.
	//
	public float ReadFloat( mod_ex mex , int reg )
	{
		ReadString( mex , reg );
		return Float.parseFloat( mex.s[0] );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write a floating point value to a register.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteFloat( mod_ex mex , int chn , int reg , float val )
	{
		return WriteString( mex , chn , reg , Float.toString( val ) );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write a floating point value to a register.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteFloat( mod_ex mex , int reg , float val )
	{
		return WriteString( mex , reg , Float.toString( val ) );
	}
	//
	//-------------------------------------------------------------------------
	// Register read/write; with and w/o channel parameter.
	//-------------------------------------------------------------------------
	//
	// Here to read a register as an string value.
	// Returns 0 if success, -1 if failure.
	//
	public String ReadReg( mod_ex mex , int chn , int reg )
	{
		ReadString( mex , chn , reg );
		return mex.s[0];
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a register as an string value.
	// Returns 0 if success, -1 if failure.
	//
	public String ReadReg( mod_ex mex , int reg )
	{
		ReadString( mex , reg );
		return mex.s[0];
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write an integer value to a register.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteReg( mod_ex mex , int chn , int reg , String val )
	{
		return WriteString( mex , chn , reg , val );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write an integer value to a register.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteReg( mod_ex mex , int reg , String val )
	{
		return WriteString( mex , reg , val );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a list of registers.
	// Returns 0 if success, -1 if failure.
	public int ReadRegList( mod_ex mex , int qty )
	{
		mex.slv_adr = cur_slv_adr;
		mex.fcn = (byte)mod_def.RD_REGS_LIST;
		mex.itm_adr = 0;
		mex.itm_qty = (short)qty;
		return MasterTx( mex );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to read a single string value.
	// Returns string if success, "ERROR" if failure.
	//
	public String ReadCfgReg( mod_ex mex , int reg )
	{
		mex.slv_adr = (byte)CONFIG_ADDRESS;
		mex.fcn = (byte)mod_def.RD_REGS_STR;
		mex.itm_adr = (short)reg;
		mex.itm_qty = 1;
		if ( MasterTx( mex ) == 0 )
			return mex.s[0];
		else
			return "ERROR";
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write a string value.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteCfgReg( mod_ex mex , int reg , String val )
	{
		mex.slv_adr = (byte)CONFIG_ADDRESS;
		mex.fcn = (byte)mod_def.WR_REGS_STR;
		mex.itm_adr = (short)reg;
		mex.itm_qty = 1;
		mex.s[0] = val;
		return MasterTx( mex );
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to write a string value.
	// Returns 0 if success, -1 if failure.
	//
	public int WriteCfgReg( mod_ex mex , int reg , int val )
	{
		mex.slv_adr = (byte)CONFIG_ADDRESS;
		mex.fcn = (byte)mod_def.WR_REGS_STR;
		mex.itm_adr = (short)reg;
		mex.itm_qty = 1;
		mex.s[0] = Integer.toString( val );
		return MasterTx( mex );
	}
    //
    //-------------------------------------------------------------------------
    //
	// Here to read an object property.
	// Property is returned in mex.s[0].
	// Returns 0 if success, -1 if failure with failure code in mex.b[0];
	//
	public int ReadObj( mod_ex mex , boolean idx_flag , int obj_instance , int obj_property , String val )
	{
        mex.slv_adr = (byte)CONFIG_ADDRESS;
        mex.fcn = (byte)mod_def.RD_OBJ;
        if ( idx_flag == true )
            mex.itm_adr = (short)0x8001;
        else
            mex.itm_adr = 1;
        mex.itm_qty = 1;
        mex.l[0] = obj_instance;
        mex.w[0] = (short)obj_property;
        if ( MasterTx( mex ) == 0 )
        {
            val = mex.s[0];
            return 0;
        }
        else
        {
            return mex.b[0];
        }
	}
    //
    //-------------------------------------------------------------------------
    //
    // Here to write an object property.
    // Property is returned in mex.s[0].
    // Returns 0 if success, -1 if failure with failure code in mex.b[0];
    //
    public int WriteObj( mod_ex mex , boolean idx_flag , int obj_instance , int obj_property , String val )
    {
        mex.slv_adr = (byte)CONFIG_ADDRESS;
        mex.fcn = (byte)mod_def.WR_OBJ;
        if ( idx_flag == true )
            mex.itm_adr = (short)0x8001;
        else
            mex.itm_adr = 1;
        mex.itm_qty = 1;
        mex.l[0] = obj_instance;
        mex.w[0] = (short)obj_property;
        mex.s[0] = val;
        if ( MasterTx( mex ) == 0 )
            return 0;
        else
            return mex.b[0];
    }
	//
	//-------------------------------------------------------------------------
	//
	public boolean StrToBool( String val )
	{
		if ( val.equals( "1" ) )
			return true;
		else
			return false;
	}
	//
	//-------------------------------------------------------------------------
	//
	public String BoolToStr( boolean val )
	{
		if ( val == true )
			return "1";
		else
			return "0";
	}
	//
	//-------------------------------------------------------------------------
	//
	public void SetProtocol( byte mod_protocol )
	{
		protocol = mod_protocol;
	}
	//
	//-------------------------------------------------------------------------
	//
	public int GetProtocol()
	{
		return protocol;
	}
	//
	//-------------------------------------------------------------------------
	//
	public void SetSlaveAddress( byte address )
	{
		cur_slv_adr = address;
	}
	//
	//-------------------------------------------------------------------------
	//
	// Here to return the current slave address.
	//
	public byte GetSlaveAddress()
	{
		return cur_slv_adr;
	}
	//
	//-------------------------------------------------------------------------
	//
	// Constructor.
	public mod_rxtx( InputStream is , OutputStream os )
	{
		// Save input and output sources.
		dis = is;
		dos = os;
		// Allocate memory for buffers.
		rxBuf = new byte[ BUFFER_SIZE ];
		xxBuf = new byte[ BUFFER_SIZE ];
		txBuf = new byte[ BUFFER_SIZE ];
	}
    //
    //-------------------------------------------------------------------------
    //
    // Here to wait for a packet.
    //
    public int ReadPacket(byte[] buf) throws IOException
    {
        int     rxlen;

        // Read a packet.
        try
        {
            rxlen = dis.read(buf);
        }
        catch (IOException e )
        {
            throw e;
        }
        // Return with length.
        return rxlen;
    }
}
