//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

#ifndef		SCREEN_HPP
#define		SCREEN_HPP

// ----------------------------------------------------------------------------
//
//		Basic screen I/O device.
//
// ----------------------------------------------------------------------------
//
class scrnDev
{
	public:

	static char	overType;	// Overtype for all screen elements.
        static char	*errorPtr;	// Error string for status display.
        int		xPos;		// X coordinate from screen home (1-80).
	int		yPos;		// Y coordinate from screen home (1-25).
        scrnDev		*next;		// Next device in chain.
        int		(*dataCheckFcn)(char *s);// Pointer to data checking function.
        int             (*preprocessFcn)( int key );// Pointer to key preprocessing function.

        // Constructor.
        //
        scrnDev( void );

        // Display entire contents of device.
        //
        virtual void show( void );

        // Position cursor at focus point.
        //
        virtual void showCursor( void );

        // Handle character input.
        //
        virtual int processChar( int c );

	// Validate data.
        //
        virtual int validate( void );
};

// ----------------------------------------------------------------------------
//
//		Edit Box
//
// ----------------------------------------------------------------------------

// Edit box structure.
//
class editBox : public scrnDev
{
	public:

        int	width;		// Number of characters in window.
        int	cursorPos;	// Cursor position in buffer.
        int	cnt;		// Number of characters currently in buffer.
        char	*buf;		// Pointer to character buffer.
        char	*filter;	// Pointer to acceptance filter.

	// Constructor.
        //
        editBox( void );

	// Initialization.
        // Allocates memory and sets initial values.
        //
        int init( int xpos , int ypos , int width );

        // Display entire contents of device.
        //
        void show( void );

        // Position cursor at focus point.
        //
        void showCursor( void );

        // Handle character input.
        // Returns:
        //      key     : Key not handled.
        //      SUCCESS : Key handled.
        //      FAILURE : Error encountered while processing key.
        //
        int processChar( int key );

	// Validate data.
        //
        int validate( void );

	// Puts string in edit box without displaying it.
        //
        void putStr( char *s );

        // Inserts a character in edit box at cursor position. Moves all characters
        // ahead of cursor; cursor position moves ahead 1 character.
        //
        void insert( char c );

        // Deletes the character to the left of the cursor. Cursor moves left one
        // character, and string collapses on right.
        //
        void backspace( void );

        // Moves cursor right one position in string.
        // Has no effect if cursor at end of string.
        //
        void rightArrow( void );

        // Moves cursor left one position in string.
        // Has no effect if cursor at beginning of string.
        //
        void leftArrow( void );
};

// ----------------------------------------------------------------------------
//
//		List Box
//
// ----------------------------------------------------------------------------

// List box structure.
// A list box is just an array of edit boxes with a character handler
// that moves between the edit boxes.
// Capable of handling more edit boxes than can be shown. Edit boxes will
// scroll up and down in fixed window.
//
class listBox : public scrnDev
{
	public:

        int		width;		// Width of display window.
        int		height;		// Height of display window.
        int		ebCnt;		// Number of edit boxes.
        int		ebTop;		// Index of edit box at top of window.
        int		ebFocus;	// Index of edit box with focus.
        int		ebLast;		// Index of first available edit box.
	editBox		*ebPtr;		// Pointer to array of edit boxes.
        int             scroll;         // TRUE if scrolling enabled.

	// Constructor.
        //
        listBox( void );

	// Initializes list box.
        // Allocates memory and sets initial values.
        //
        int init( int xpos , int ypos , int width , int height , int cnt );

        // Display entire contents of device.
        //
        void show( void );

        // Position cursor at focus point.
        //
        void showCursor( void );

        // Handle character input.
        // Returns:
        //      key     : Key not handled.
        //      SUCCESS : Key handled.
        //      FAILURE : Error encountered while processing key.
        //
        int processChar( int key );

        // Adds a string at the first open edit box within a list box.
        //
        void addStr( char *s );

        // Here to 'delete' the current line in a list box.
        // This moves the contents of all higher edit boxes down one index.
        //
        void deleteLine( void );

        // Here to 'insert' a blank line in a list box.
        // This moves the contents of all higher edit boxes up one index.
        //
        void insertLine( void );
};

// ----------------------------------------------------------------------------
//
//              Overall screen control.
//
// ----------------------------------------------------------------------------

// Structure that holds pointers to screen display elements.
//
class screen
{
	public:

	int		count;		// Number of display devices.
	scrnDev		*first;		// First device in linked list.
        scrnDev		*focus;		// Device with focus.
        int		otXpos; 	// Column for overtype indication.
        int		otYpos;		// Line for overtype indication.

	// Constructor.
        //
        screen( void );

        // Here to refresh an entire screen.
        //
        void show( void );

        // Position cursor at focus point.
        //
        void showCursor( void );

        // Here to process characters for screen.
        // Returns:
        //      key     : Key not handled.
        //      SUCCESS : Key handled.
        //      FAILURE : Error encountered while processing key.
        //
        int processChar ( int key );

        // Here to add a display element to a screen.
        // Returns SUCCESS or FAILURE.
        //
        int addWidget( scrnDev *sdp );
};

#endif

