//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

#ifndef	PCTIMER_H
#define	PCTIMER_H

// Timer object operating modes.
//
// TM_PULSE:	Timer operation stops at timeout.
// TM_RETRIG:	Timer is "retriggered" at timeout.
//
#define		TM_PULSE	0		// Default.
#define		TM_RETRIG	1

// Timer structure.
//
typedef struct timer
{
	char		mode;  		// Operating mode.
	char		status;	 	// Time-out status.
	long		preset;		// Preset value.
	long		current; 	// Current value.
	struct timer	*nextTimer;	// Pointer to next timer in chain.
	void		(*isr)( void );	// Pointer to routine to execute
					// when status is set. Should be a
					// REAL SHORT routine.
} timer;

// Global timer variables.
//
			// Flag: when '1', the system timer will be called at
                        // the 'regular' inerval.
                        // Defaults to '1'.
extern int		sysTimer;

			// Allows interrupts within timer interrupt when '1'.
                        // Defaults to '0'.
extern int		intEnable;

			// Incremented once each interrupt, if timer is hooked.
extern unsigned long	tmrTics;

#ifdef          __cplusplus
extern "C" {
#endif

// Timer functions.
//
			// Allocates storage and initializes timer.
			// Returns NULL if out of memory.
timer	*tmrInit	( void );

			// Removes timer from chain and releases storage.
void	tmrDestroy	( timer *tp );

			// Starts timer with 'val' counts.
void	tmrStart	( timer *tp , long val );

			// Starts timer with 'val' seconds.
                        // This routine evaluates the time period to determine
                        // the number of counts.
void	tmrStartSec	( timer *tp , long val );

			// Stops timer.
void	tmrStop		( timer *tp );

			// Loads new preset value (for retrigerable).
void	tmrLoadPreset	( timer *tp , long val );

			// Loads new curretn value.
void	tmrLoadCurrent	( timer *tp , long val );

			// Sets timer mode (TM_PULSE or TM_RETRIG).
void	tmrSetMode 	( timer *tp , char timerMode );

			// Sets pointer to SHORT routine that executes when
			// timer times out.
void	tmrSetIsr	( timer *tp , void (*timerIsr)( void ) );

			// Returns '1' when timer has timed out; '0' otherwise.
int	tmrFlag		( timer *tp );

// ----------------------------------------------------------------------------
// Routines to install & remove timer interrupt function.
// ----------------------------------------------------------------------------
//
// This routine installs the timer object system timer tic handler.  It is
// installed at the "top of the chain" of any other system timer tic
// handlers (for PC timer 0).  It can increase the frequency of system
// timer tics.  It calls the previous timer tic handler on (approximately)
// the same rate as the PC standard (18.2 tics/second).
// Value 'newMult' specifies new timer operating rate as an integer
// multiple of the basic PC rate of 18.2 tics/second.
// Sample values and their associated time periods:
//
//	1:	18.2 tics/second	54.925 mS/tic (PC standard)
//	2:	36.4 tics/second	27.463 mS/tic
//	55:	1001 tics/second        998 uS/tic
//	110:	2002 tics/second	499 uS/tic
//
void tmrHook( int newMult );

// This routine MUST be called when program terminates (if the timer interrupt
// is hooked) to remove the timer tic interrupt handler from the "chain".
// Interrupt handler goes away when program terminates, and if the interrupt
// vector is still set to our (now nonexistent) routine the PC goes off into
// the weeds.
//
void tmrUnhook( void );

#ifdef          __cplusplus
}
#endif

#endif	PCTIMER_H
