//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

/*
	PC Timer.

        Revision History.

1.00	2/15/00

	First issue of this version.
        'tmrInit()' returns a pointer to a NEW timer object.
        Will operate in BIOS mode if 'tmrHook()' not called (no ints).

1.01	3/10/00

	Changed 'tmrRemove()' to 'tmrDestroy()'.
        Now CLI and clear flag in 'tmrStop()'.

1.02	3/18/0

	Added 'tmrStartSec()' to do timing in seconds.
*/

#include	"compiler.h"
#if		MICROSOFT_COMPILER
#include	<malloc.h>
#else
#include	<alloc.h>
#endif
#include	<dos.h>
#include	<conio.h>
#include	<stdlib.h>
#include	<bios.h>
#include	"pctimer.h"

//-----------------------------------------------------------------------------
// Constants.
//-----------------------------------------------------------------------------

// Timer interrupt number.
//
#define		TMR_INT		0x08	// System timer interrupt vector #.
					// (Interrupt 0x1c is called by 0x08)

// Timer chip addresses and control bytes.
//
#define		TMR_0		0x40
#define		TMR_CTRL	0x43
#define		TMR_0_LD	0x36

// 8259 Interrupt controller data.
//
#define		PIC_EOI		0x20		// EOI port (W).
#define		EOI		0x20		// Normal EOI command.

//-----------------------------------------------------------------------------
// Global data.
//-----------------------------------------------------------------------------

int		sysTimer = 1;		// Call system timer when '1'.

int		intEnable = 0;		// Enable interrupts within timer
					// interrupt when '1'.

unsigned long	tmrTics;		// Incremented once each interrupt.

//-----------------------------------------------------------------------------
// Local data.
//-----------------------------------------------------------------------------

#ifdef	__cplusplus
static void interrupt (far *oldTimer)(...);// Original interrupt handler.
#else
static void interrupt (far *oldTimer)();// Original interrupt handler.
#endif

static int	timerHooked = 0;	// Set if system timer hooked.

static timer	*timerChain = 0;	// Pointers to chain of timer
					// objects.

static int	timerMult = 1;		// Number of "hits" before
					// OldTimer() is called.

static int  	timerCount = 0;		// Current "hit" count.

//-----------------------------------------------------------------------------
// Timer tic interrupt routine.
// This routine is placed at the top of the "chain" of any previously
// installed timer tic interrupt handlers.
//-----------------------------------------------------------------------------

#ifdef	__cplusplus
static void interrupt far newTimer(...)
#else
static void interrupt far newTimer()
#endif
{
	timer	*t = timerChain;

	// Increment timer tics.
        ++tmrTics;

	// Enable interrupts?
        if ( intEnable )
        {
	   __asm	sti
        }
	// Do GP timer logic.
	while ( t )
	{
	   // If timer current value not '0'.
	   if ( t->current != 0 )
	   {
	      // If timer goes to '0'.
	      if ( --t->current == 0 )
	      {
		 // Set 'done' flag.
		 t->status = 1;
		 // Execute the 'isr' routine if pointer not null.
		 if ( t->isr != 0 ) (*t->isr)();
		 // Reload current value if retriggerable timer.
		 if ( t->mode == TM_RETRIG )
		    t->current = t->preset;
	      }
	   }
	   // Point to next timer in list.
	   t = t->nextTimer;
	}

	// Calling the system timer function?
        if ( sysTimer )
        {
           // Call system function when needed.
           // It will take care of EOI.
           if ( ++timerCount >= timerMult )
           {
              timerCount = 0;
              (*oldTimer)();
           }
           else
              // Else do the EOI.
              outportb( PIC_EOI , EOI );
        }
	else
	{
	   // Do the EOI.
	   outportb( PIC_EOI , EOI );
	}
}

//-----------------------------------------------------------------------------
// Global routines.
//-----------------------------------------------------------------------------

// Here to install the timer interrupt handler.
//
void tmrHook( int newMult )
{
	int	newCount;
	long	count = 0x10000L;

	newCount = (int)( count / ( timerMult = newMult ) );
	oldTimer = getvect( TMR_INT );
	setvect( TMR_INT , newTimer );
	__asm	pushf
	__asm	cli
	outportb( TMR_CTRL , TMR_0_LD );
	outportb( TMR_0 , newCount & 0xff );
	outportb( TMR_0 , (unsigned)newCount >> 8 );
	__asm	popf
	// Set flag.
	timerHooked = 1;
}

// Here to remove the timer interrupt handler.
//
void tmrUnhook( void )
{
	__asm	pushf
	__asm	cli
	setvect( TMR_INT , oldTimer );
	outportb( TMR_CTRL , TMR_0_LD );
	outportb( TMR_0 , 0 );
	outportb( TMR_0 , 0 );
	__asm	popf
	// Reset flag.
	timerHooked = 0;
}

//-----------------------------------------------------------------------------
// Timer support functions.
//-----------------------------------------------------------------------------

static void insertTimer( timer *newTmr )
{
	timer	*tp = timerChain;

	__asm	pushf
	__asm	cli
	if ( tp == 0 )
	{
	   timerChain = newTmr;
	   newTmr->nextTimer = 0;
	   __asm	popf
	   return;
	}
	for (;;)
	{
	   if ( tp->nextTimer == 0 )
	   {
	      tp->nextTimer = newTmr;
	      newTmr->nextTimer = 0;
	      break;
	   }
	   tp = tp->nextTimer;
	}
	__asm	popf
}

//-----------------------------------------------------------------------------
// Timer member functions.
//-----------------------------------------------------------------------------

// Initializer.
//
struct timer *tmrInit( void )
{
	struct timer 	*tp;

	// Allocate storage for timer.
	if ( ( tp = (timer *)malloc( sizeof( struct timer ) ) ) != NULL )
	{
	   tp->mode = TM_PULSE ;
	   tp->status = 0;
           tp->preset = tp->current = 0;
	   tp->isr = 0;
	   insertTimer( tp );
	}
	// Return with pointer to timer object.
	return tp;
}

// Destructor.
//
void tmrDestroy( timer *t )
{
	timer	*tp = timerChain , *tp2;

	// Check pointer.
        if ( t == 0 ) return;
	// Disable interrupts.
	__asm	pushf
	__asm	cli
        // If timer is root timer.
	if ( tp == t )
	{
           // Just reload root timer pointer.
	   timerChain = tp->nextTimer;
           // And release memory.
           free( t );
           // Restore interrupts.
	   __asm	popf
           // And return.
	   return;
	}
        // Loop to find 't'.
	for (;;)
	{
           // If the next timer in chain is the one.
	   if ( tp->nextTimer == t )
	   {
              // Get pointer to it.
	      tp2 = tp->nextTimer;
              // Adjust chain of pointers.
	      tp->nextTimer = tp2->nextTimer;
              // Exit loop.
	      break;
	   }
           // Else point to next timer.
	   tp = tp->nextTimer;
	}
        // Release memory.
	free( t );
	// Restore interrupts.
	__asm	popf
}

// Loads timer preset and current values with 'val'.
//
void tmrStart( timer *tp , long val )
{
	// Check pointer.
        if ( tp == 0 ) return;

	if ( timerHooked )
        {
           __asm	pushf
           __asm	cli
           tp->preset = tp->current = val;
           tp->status = 0;
           __asm	popf
	}
        else
        {
           tp->preset = val;
           tp->current = biostime( 0 , 0L ) + val;
        }
}

// Loads timer preset and current values to get 'val' seconds.
//
void tmrStartSec( timer *tp , long val )
{
	// Check pointer.
        if ( tp == 0 ) return;
        // Only recognize certain divisors.
        switch( timerMult )
        {
        // Standard 18.2 times/sec.
        case 1:
           val *= 18;
           break;
        // Milliseconds.
        case 55:
           val *= 1000;
           break;
        // Half milliseconds.
        case 110:
           val *= 2000;
           break;
        // Quarter milliseconds.
        case 220:
           val *= 4000;
           break;
        default:
           // Just return if unknown divisor.
           return;
        }
	//
	if ( timerHooked )
        {
           __asm	pushf
           __asm	cli
           tp->preset = tp->current = val;
           tp->status = 0;
           __asm	popf
	}
        else
        {
           tp->preset = val;
           tp->current = biostime( 0 , 0L ) + val;
        }
}

// Stops timer operation.
//
void tmrStop( timer *tp )
{
	// Check pointer.
        if ( tp == 0 ) return;

        __asm	pushf
        __asm	cli
        tp->preset = tp->current = 0;
        tp->status = 0;
        __asm	popf
}

// Loads timer preset value with 'val'.
//
void tmrLoadPreset( timer *tp , long val )
{
	// Check pointer.
        if ( tp == 0 ) return;

	__asm	pushf
	__asm	cli
	tp->preset = val;
	__asm	popf
}

// Loads timer current value with 'val'.
//
void tmrLoadCurrent( timer *tp , long val )
{
	// Check pointer.
        if ( tp == 0 ) return;

	__asm	pushf
	__asm	cli
	tp->current = val;
	__asm	popf
}

// Sets timer operating mode.
//
void tmrSetMode( timer *tp , char tmode )
{
	// Check pointer.
        if ( tp == 0 ) return;

	__asm	pushf
	__asm	cli
	tp->mode = tmode;
	__asm	popf
}

// Sets timer "interrupt service routine".
//
void tmrSetIsr( timer *tp , void (*timerIsr)( void ) )
{
	// Check pointer.
        if ( tp == 0 ) return;

	__asm	pushf
	__asm	cli
	tp->isr = timerIsr;
	__asm	popf
}

// Here to see if timer has timed out.
// Return status value; reset status if set.
//
int tmrFlag( timer *tp )
{
	long	ttime;

	// Check pointer.
        if ( tp == 0 ) return 0;

	if ( timerHooked )
        {
           if ( tp->status )
           {
              tp->status = 0;
              return 1;
           }
        }
        else
        {
           if ( tp->current == 0 ) return 0;
           ttime = biostime( 0 , 0L );
           if ( ttime > tp->current )
           {
              if ( tp->mode == TM_PULSE )
                 tp->current = 0;
              else
                 tp->current = ttime + tp->preset;
              return 1;
           }
        }
        return 0;
}

