//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

#ifndef		COMPORT_H
#define		COMPORT_H

// Return values.
//
#define		TIMEOUT		-1
#define		NULL_CHAR	-1

// Com port.
// Use these for 'port' in 'comInit()'.
//
#define		COM1		0
#define		COM2		1
#define		COM3		2
#define		COM4		3

// Data format.
// Use these for 'format' in 'comInit()'.
// You can combine flags: _7BIT + PARITY + EVEN , etc.
//
#define		_7BIT		0x01
#define		_8BIT		0x02
#define		_2STOP		0x04
#define		PARITY		0x08
#define		EVEN		0x10

// Baud rate.
// Use these for 'baud' in 'comInit()'.
//
#define		_1200		0
#define		_2400		1
#define		_4800		2
#define		_9600		3
#define		_19200		4
#define		_38400		5
#define		_115200		6

// Line status register bit masks.
//
#define		LSR_RXRDY	0x001		// Data ready.
#define		LSR_OVERRUN	0x002		// Overrun error.
#define		LSR_PARITY	0x004		// Parity error.
#define		LSR_FRAMING	0x008		// Framing error.
#define		LSR_BREAK	0x010		// Break interrupt.
#define		LSR_TXHRE	0x020		// Transmit holding empty.
#define		LSR_TXSRE	0x040		// Transmit SR empty.
#define		TX_EMPTY_	LSR_TXHRE + LSR_TXSRE	// Both empty.

// Modem status register bit masks.
//
#define		MSR_DCTS	0x001		// Delta clear to send.
#define		MSR_DDSR	0x002		// Delta data set ready.
#define		MSR_DRI		0x004		// Delta ring indicator.
#define		MSR_DCDT	0x008		// Delta carrier detect.
#define		MSR_CTS		0x010		// Clear to send.
#define		MSR_DSR		0x020		// Data set ready.
#define		MSR_RI		0x040		// Ring indicator.
#define		MSR_CDT		0x080		// Carrier detect.

// Modem control register bit masks.
//
#define		MCR_DTR		0x01		// DTR bit.
#define		MCR_RTS		0x02		// RTS bit.
#define		MCR_IEN		0x08		// IEN bit.
#define		MCR_LUP		0x10		// LOOP bit.

// Character definitions.
//
#define		NULL_CHAR	-1
#define		XON		0x11
#define		XOFF		0x13
//
// Initialization failure codes.
// If 'comInit()' fails, one of these will be in the 'comInitStatus' global variable.
//
#define		INIT_OK		0	// No error.
#define		INIT_BADPORT	1	// Bad com port number.
#define		INIT_MALLOC	2	// Can't allocate memory.
//
extern int	comInitStatus;		// Results of 'comInit()'.

// Communication port structure.
//
typedef struct comPort
{
	// User flags (public).
        char	bufferChar;	// Set to buffer chars in poll mode.
        char	xCtrl;		// Set to use xon/xoff flow control.
        char	ctsCtrl;	// Set to use cts flow control.
	// Comport flags & data (private).
	char	initPoll;	// True if 'comInitPoll()' was used to init.
        char	irqUsed;	// Set if interrupts used.
	int	port;		// Base address of 8250.
	char	msrStat;	// MSR register status.
	char	rxMode;		// Receive operating mode.
	char	txMode;		// Transmit operating mode.
        int	oldImrMask;	// Original PIC IMR mask.
	struct timer	*tmr;	// Timer for rx, tx operations.
	#ifdef __cplusplus
	void interrupt (far *oldHandler)( ... ); // Original interrupt handler.
        #else
	void interrupt (far *oldHandler)();
        #endif

	//
        int	rxSize;		// Size of rx buffer (chars).
	int	rxEchoCnt;	// Number of characters to receive if echo on.
	int	rxCount;	// Number of chars in rx_buffer.
	int	rxIn;		// Offset for putting chars in rx buffer.
	int	rxOut;		// Offset for taking chars out of rx buffer.
	//
	int	txSize;		// Size of tx buffer (chars).
	char	txRunning;	// Transmit operation in process.
	int	txCount;	// Number of chars in tx_buffer.
	int	txIn;		// Offset for putting chars in tx buffer.
	int	txOut;		// Offset for taking chars out of tx buffer.
	//
	char	*rxBuf;		// Pointer to receive buffer.
	char	*txBuf;		// Pointer to transmit buffer.
	//
} comPort;

#ifdef  __cplusplus
extern "C" {
#endif

	// Initializes com port; does format & baudrate setup.
        // Must be called before any other functions.
        // Returns pointer to com port if success.
        // Return NULL pointer if failure not enough memory, no chip, etc..
        // If failure, 'comInitStatus' gives reason.
        //
	comPort *comInit( int irqUsed , int port , int format , int baud , int rxSize , int txSize );

	// Initializes data areas and port numbers, but does
        // not access uart. Used to init for access to an
        // already set-up uart in polled mode (when console redirection
        // used and you don't want to screw with the chip).
        //
	comPort *comInitPoll( int port , int rxSize , int txSize );

	// Com port destructor.
        //
	void comDestroy( comPort *cp );

        // Service com port in polled mode.
        // Polls RX and puts characters in buffer.
        // Polls TX ready and xmits chars in buffer.
        //
        void comServicePoll( comPort *cp );

	// Sets data transmission format.
        //
	void comSetFormat( comPort *cp , int format );

	// Sets baudrate.
        //
	void comSetBaud( comPort *cp , int baud );

	// Sends a single character.
        // Places character in TX buffer in polled mode if 'bufferChar'
        // is true.
        //
	int comSendC( comPort *cp , char c );

	// Sends a null terminated string.
        //
	int comSendS( comPort *cp , char *s );

	// Sends 'cnt' characters.
        //
	int comSendSI( comPort *cp , char *s , int cnt );

	// Sends 'cnt' characters out an RS485 two wire connection; uses
        // RTS for TX enable.
        //
	int comSend485( comPort *cp , char *s , int cnt );

	// Returns a character if one in buffer.
	// Returns -1 if no characters available.
        //
	int comRecv( comPort *cp );

	// Returns a character if one received within 'time' seconds.
	// Returns -1 if no character received in time.
        //
	int comRecvT( comPort *cp , int time );

	// Returns SUCCESS if 'cnt' characters received within 'time' seconds.
	// Returns FAILURE if specified number of characters not received
        // within time allowed.
        //
	int comRecvTC( comPort *cp , int time , int cnt );

	// Puts a character back into the receive
	// buffer (as the next char to be received).
        //
	void comUngetC( comPort *cp , char c );

	// Returns the number of characters in the input buffer in interrupt
        // mode or in polled mode if 'bufferChar' is true.
        // Returns non-zero if there is a character waiting in the RX buffer
        // in polled mode when 'bufferChar' is false.
        //
	int comInStat( comPort *cp );

        // Flushes RX buffer.
        //
        void comFlushRx( comPort *cp );

	// Returns the number of characters remaining to be sent
        // in the output buffer in interrupt mode or in polled mode
        // if 'bufferChar' is true.
        // Returns non-zero if character can be sent in polled mode when
        // 'bufferChar' is false.
        //
	int comOutStat( comPort *cp );

	// Returns status of LSR register.
        //
	int comLineStatus( comPort *cp );

	// Returns status of MSR register.
        //
	int comModemStatus( comPort *cp );

	// Selectively turns modem control bits on.
        //
	void comModemControlOn( comPort *cp , char c );

	// Selectively turns modem control bits off.
        //
	void comModemControlOff( comPort *cp , char c );

	// Aborts transmission in progress.
        //
	void comTxAbort( comPort *cp );

#ifdef  __cplusplus
}
#endif

#endif		COMPORT_H
