//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

#ifndef		CAN_DRIV_H
#define		CAN_DRIV_H

//              REVISION 1.05
//              6/6/00

// Basics.
//
typedef	unsigned char	uchar;
typedef unsigned short	ushort;
typedef unsigned long	ulong;
//
#define		SUCCESS		0
#define		FAILURE		1
#define		TRUE		1
#define		FALSE		0
//
//------------------------------------------------------------------------------
//	SJA1000 Register and bitmask definitions.
//------------------------------------------------------------------------------
//
// CAN registers; PELICAN mode.
// Must be used with a pointer to a 'canObject' ('cp').
//
#define		MOD_REG		( cp->port + 0 )	// Mode.
#define		CMR_REG		( cp->port + 1 )	// Command.
#define		SR_REG		( cp->port + 2 )	// Status.
#define		IR_REG		( cp->port + 3 )	// Interrupt status.
#define		IER_REG		( cp->port + 4 )	// Interrupt enable.
							// 05 reserved.
#define		BTR0_REG	( cp->port + 6 )	// Bus timing 0.
#define		BTR1_REG	( cp->port + 7 )	// Bus timing 1.
#define		OCR_REG		( cp->port + 8 )	// Output control.
							// 09 test
			                                // 10 reserved
#define		ALC_REG		( cp->port + 11 )	// Arbitration lost capture.
#define		ECC_REG		( cp->port + 12 )	// Error code capture.
#define		EWLR_REG	( cp->port + 13 )	// Error warning limit.
#define		RXERR_REG	( cp->port + 14 )	// RX error counter.
#define		TXERR_REG	( cp->port + 15 )	// TX error counter.
#define		TXB_REG		( cp->port + 16 )	// TX buffer.
#define		RXB_REG		( cp->port + 16 )	// RX buffer.
#define		ACR0_REG	( cp->port + 16 )	// Acceptance code 0.
#define		ACR1_REG	( cp->port + 17 )	// Acceptance code 1.
#define		ACR2_REG	( cp->port + 18 )	// Acceptance code 2.
#define		ACR3_REG	( cp->port + 19 )	// Acceptance code 3.
#define		AMR0_REG	( cp->port + 20 )	// Acceptance mask 0.
#define		AMR1_REG	( cp->port + 21 )	// Acceptance mask 1.
#define		AMR2_REG	( cp->port + 22 )	// Acceptance mask 2.
#define		AMR3_REG	( cp->port + 23 )	// Acceptance mask 3.
#define		RMC3_REG	( cp->port + 29 )	// RX packet counter.
#define		RBSA_REG	( cp->port + 30 )	// RX buffer start address.
#define		CDR_REG		( cp->port + 31 )	// Clock divide.
#define		IRAM_REG	( cp->port + 32 )	// Internal ram.
//
//------------------------------------------------------------------------------
//
// Mode register bitmasks.
//
#define		RM_BIT		0x01		// Reset.
#define		LOM_BIT		0x02		// Listen only.
#define		STM_BIT		0x04		// Self test.
#define		AFM_BIT		0x08		// Acceptance filter.
#define		SM_BIT		0x10		// Sleep.
//
//------------------------------------------------------------------------------
//
// Command register bitmasks.
//
#define		TR_BIT		0x01		// Transmission request.
#define		AT_BIT 		0x02		// Abort transmission.
#define		RRB_BIT		0x04		// Release receive buffer.
#define		CDO_BIT		0x08		// Clear data overrun.
#define		SRR_BIT		0x10		// Self reception request.
//
//------------------------------------------------------------------------------
//
// Status register bitmasks.
//
#define		RBS_BIT		0x01		// Receive buffer (status).
#define		DOS_BIT		0x02		// Data overrun.
#define		TBS_BIT		0x04		// Transmit buffer.
#define		TCS_BIT		0x08		// Transmission complete.
#define		RS_BIT		0x10		// Receive.
#define		TS_BIT		0x20		// Transmit.
#define		ES_BIT		0x40		// Error.
#define		BS_BIT		0x80		// Bus.
//
//------------------------------------------------------------------------------
//
// CAN Interrupt bitmasks; used for interrupt enable & status.
//
#define		RI_BIT		0x01		// Receive.
#define		TI_BIT 		0x02		// Transmit.
#define		EI_BIT		0x04		// Error warning.
#define		DOI_BIT		0x08		// Data overrun.
#define		WUI_BIT		0x10		// Wake-up.
#define		EPI_BIT		0x20		// Error Passive.
#define		ALI_BIT		0x40		// Arbitration lost.
#define		BEI_BIT		0x80		// Bus error.
//
//------------------------------------------------------------------------------
//
// RX/TX Frame information bitmasks.
//
#define		FF_BIT		0x80		// Frame format.
#define		RTR_BIT		0x40		// Remote Transmission Request.
#define		DLC_BITS	0x0f		// Data length bits.
//
//------------------------------------------------------------------------------
//      CAN driver structures and functions.
//------------------------------------------------------------------------------
//
// Acceptance filters.
// Used in 'type' data element of 'canFilter' structure below.
//
// Standard frame/single filter.
// One 11 bit filter with RTR and two data bytes.
//
#define		STD_SNGL	0
//
// Standard frame/dual filter.
// Two 11 bit filters with RTR and one data byte.
//
#define		STD_DUAL	1
//
// Extended frame/single filter.
// One 29 bit filter with RTR.
//
#define		EXT_SNGL	2
//
// Extended frame/dual filter.
// Two 16 bit filters without RTR.
//
#define		EXT_DUAL	3
//
//-------------------------------
//
// CAN baud rates.
// Passed as parameter to 'canSetBaud()'.
//
#define		BAUD_125KB	1
#define		BAUD_250KB	2
#define		BAUD_500KB	3
#define		BAUD_1000KB	4
//
// CAN status bits.
// Bitfield definitions for value returned by 'canStatus()'.
//
#define		STAT_RXRDY	0x01	// Message waiting in RX buffer.
#define		STAT_OVERRUN	0x02	// Receive data overrun.
#define		STAT_TXRDY	0x04	// TX buffer will accept packet.
#define		STAT_TXCOMPLETE	0x08	// TX buffer empty and TX complete.
#define		STAT_ERROR	0x10	// Chip reports error condition.
#define		STAT_BUSOFF	0x20	// Chip is in bus-off condition.
//
#define		MAX_ERROR	16	// Size of error log.
//
// Time function codes.
// These are used by 'canTime()' to determine which function to perform.
//
#define		TIME_RESET	0x01	// Resets time stamp to zero.
					// Used to define a starting point for time
					// stamps.
#define		TIME_TRIG	0x02	// Sets trigger so that next received packet
					// gets a time stamp of '0'.
                                        // Used to start a sequence of time stamps
                                        // starting from '0'.
//
// Initialization failure codes.
// If 'canInit()' fails, one of these will be in the 'canInitStatus' global variable.
//
#define		INIT_OK		0	// No error.
#define		INIT_NOCHIP	1	// Can't find/reset chip.
#define		INIT_BADIRQ	2	// Bad irq number.
#define		INIT_MALLOC_CP	3	// Can't allocate memory for object.
#define		INIT_MALLOC_TX	4	// Can't allocate memory for tx buffer.
#define		INIT_MALLOC_RX	5	// Can't allocate memory for rx buffer.
//
extern short	canInitStatus;		// Results of 'canInit()'.
//
//------------------------------------------------------------------------------
//
// Structure used to store rx/tx packets in memory buffer.
// Used internally by the 'canObject' functions.
//
typedef struct rawCanBuffer
{
	ulong	time;		// Time stamp.
	uchar  	info; 		// EFF, RTR, DLC bits.
        uchar  	data[12]; 	// Identifier(2/4) + data(0-8)
};
//
//------------------------------------------------------------------------------
//
// Structure used for receive and transmit operations.
// This is what the user will work with.
//
typedef struct canBuffer
{
	ulong  	time;		// Time stamp returned by RX service.
	char   	type;		// 0 = Standard Frame, 1 = Extended Frame.
	long   	id;		// Identifier: 11 or 29 bits.
        char   	rtr;		// Remote Transmission Request bit.
	char   	cnt;		// Number of bytes in packet.
        uchar  	data[8];	// RX/TX data.
};
//
//------------------------------------------------------------------------------
//
// Structure used to set acceptance filters.
//
typedef struct canFilter
{
	char	type;	  	// Type of filter; STD_SNGL, etc.
                                //
				// Filter 1.
                                //
	long	idMask1;	// Identifier mask.
        long	idCode1;	// Identifier matching value.
	char	rtrMask1;	// RTR bit mask(s).
        char	rtrCode1;	// RTR bit value(s).
                                //
        			// Filter 2.
				//
	long	idMask2;	// Identifier mask.
        long	idCode2;	// Identifier matching value.
        char	rtrMask2;	// RTR bit mask(s).
        char	rtrCode2;	// RTR bit value(s).
				//
        			// Data Filter.
                                //
        uchar	dataMask[2];	// Data byte(s) mask.
        uchar	dataCode[2];	// Data byte(s) value.
};
//
//------------------------------------------------------------------------------
//
// CAN object structure.
//
typedef struct canObject
{
	char	inService;			// Set when initialized.
        char	irqUsed;			// Set when using interrupts.
	char	irq;				// IRQ number (0-15).
	short	port;				// Base address of SJA1000.
        short	oldImrMask;			// Original PIC IMR mask.
	#ifdef __cplusplus			// Original interrupt handler.
	void (interrupt far *oldHandler)( ... );
	#else
	void (interrupt far *oldHandler)();
	#endif
	uchar	modeReg;			// Mode control bits.
        //
	char	txRunning;			// Transmit operation in process.
	uchar	errorFlags;			// Accumulated error bits.
	ushort	errorCnt;			// Number of error ints.
	ushort	errorIdx;			// Error log index.
	uchar	errorLog[MAX_ERROR];		// Error log.
	char	rxOverrun;			// Set if overrun.
	char	timeTrig;			// Set if next receive int to be 'zero' time.
	long	timeZero;			// 'Zero point' for time stamps.
	//
	ushort	rxBufSize;			// Max number of receive 'canBuffer'.
	ushort	rxCount;			// Number of 'canBuffer' in rx buffer.
	ushort	rxIn;				// Index for putting packets in rx buffer.
	ushort	rxOut;				// Index for taking packets out of rx buffer.
	struct	rawCanBuffer huge	*rxBuf;// Pointer to receive buffer.
	//
	ushort	txBufSize;			// Max number of transmit 'canBuffer'.
	ushort	txCount;			// Number of 'canBuffer' in tx buffer.
	ushort	txIn;				// Index for putting packets in tx buffer.
	ushort	txOut;				// Index for taking packets out of tx buffer.
	struct	rawCanBuffer huge	*txBuf;	// Pointer to transmit buffer.
};

#ifdef          __cplusplus
extern "C" {
#endif
//
//------------------------------------------------------------------------------
//
// CAN object function prototypes.
//
//
				// Initializes CAN chip and data structure(s).
                                // Allocates memory for 'canObject' and rx/tx
                                // buffers.
                        	// Returns pointer to initialized CAN object
                                // if success.
				// Return NULL pointer if failure.
                        	//
struct canObject *canInit	( short irq ,
				  ushort ioAddress ,
                                  short baudrate ,
                                  struct canFilter *filter,
                                  short rxBufSize ,
                                  short txBufSize );

                                // Alternate initialization function; is passed
                                // explicit parameters for bit timing registers.
                                // Performs same functions as 'canInit()'.
                        	// Returns pointer to initialized CAN object
                                // if success.
				// Return NULL pointer if failure.
                        	//
struct canObject *canInitBtr	( short irq ,
				  ushort ioAddress ,
                                  short sjw ,
                                  short prescale,
                                  short tseg1,
                                  short tseg2,
                                  short sampling,
                                  struct canFilter *filter,
                                  short rxBufSize ,
                                  short txBufSize );

			// Disables CAN chip, releases allocated memory,
			// and unhooks interrupt.
                        //
void	canDestroy	( struct canObject *cp );

			// Puts CAN chip in reset mode
                        // Returns SUCCESS or FAILURE if chip can't be reset.
			//
short	canReset	( struct canObject *cp );

			// Brings CAN chip out of reset mode.
			// Should be used after setup, or to bring a chip
                        // back online after a bus-off condition.
                        // Returns SUCCESS or FAILURE if chip can't be brought
                        // out of reset.
                        //
short	canEnable	( struct canObject *cp );

                        // Sets baud rate.
			// Returns SUCCESS or FAILURE if invalid baud rate.
                        //
short	canSetBaud	( struct canObject *cp , short baudrate );

                        // Initializes bit timing registers. Used to explicity
                        // set up baud rate and bit timing parameters.
			// Returns SUCCESS or FAILURE if invalid baud rate.
                        //
short	canSetBtr	( struct canObject *cp , short sjw ,
                                                 short prescale,
                                                 short tseg1,
                                                 short tseg2,
                                                 short sampling );

			// Sets acceptance filter for received packets.
                        // Returns SUCCESS or FAILURE if invalid filter type.
                        //
short	canSetFilter	( struct canObject *cp , struct canFilter *filter );

			// Transmits a packet.
                        // Returns SUCCESS or FAILURE if buffer full.
                        //
short	canSend		( struct canObject *cp , struct canBuffer *txBuf );

			// Receives a packet; fills 'rxBuf' with data.
                        // Returns SUCCESS if packet waiting and data transferred
                        // to 'rxBuf'.
                        // Returns FAILURE if no packet available.
                        //
short	canRecv		( struct canObject *cp , struct canBuffer *rxBuf );

                        // Used to put chip in listen only mode. Chip will receive
                        // packets, but will not acknowledge.
                        // Pass TRUE to put the chip in listen only mode; pass
                        // FALSE to put the chip in normal mode.
                        // Returns SUCCESS or FAILURE.
                        //
short   canListenOnly   ( struct canObject *cp , short onOff );

			// Returns status informaton.
			// Refer to STAT_XXX #defines above.
			//
ushort	canStatus	( struct canObject *cp );

			// Returns, and then clears, accumulated error status
			// bits.
                        //
ushort	canError	( struct canObject *cp );

			// Discard any packets in tx buffer waiting to be sent.
                        //
void	canTxPurge	( struct canObject *cp );

			// Discard any packets in rx buffer waiting to be read.
                        //
void	canRxPurge	( struct canObject *cp );

			// Recovers from bus-off (or other error condition).
                        // Resets the chip, purges tx & rx packets, and takes the
                        // chip out of reset.
                        //
void	canRecover	( struct canObject *cp );

			// Performs timer functions needed for time stamp.
			// 1 = Reset time stamp to '0'.
			// 2 = Reset time stamp to '0' when next packet
			//     is received.
			// See TIME_XXX #defines above.
			//
void	canTime		( struct canObject *cp , short funtion );


#ifdef          __cplusplus
}
#endif

#endif
