//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

/*
	CAN Driver.

        Revision History.

1.00	First issue; sent to CCI and noted as preliminary.

1.01	2/15/00

	Bit masking error fixed in 'intService' regarding clearing of error
	bits in 'intReg' variable.

1.02	2/16/00

	canStatus() return SUCCESS instead of 'retval'.

	_DBUG: Changed irq setup to only enable rx & tx interrupts.

1.03	2/17/00

	Messed up 'time' in 'can_driv.c'. Going to 'ulong' and 17 byte raw
	buffers?

1.04	3/10/00	RCW

	Improved check for presence of chip in 'canInit()'. Now use timer when
        checking for chip.

1.05    6/6/00  RCW

        Added 'canInitBtr()' , 'canSetBtr()', and 'canListenOnly()'.
        Changed references from 'message' to 'packet'.
*/
#include	"compiler.h"
#if		MICROSOFT_COMPILER
#include	<malloc.h>
#else
#include	<alloc.h>
#endif
#include	<dos.h>
#include	<conio.h>
#include	<stdlib.h>
#include	"can_driv.h"
#include	"pctimer.h"
//
//---------------------------------------------------------------------------
//
// PC Interrupt controller ports (PIC).
//
#define		PCL_OCW1	0x21		// PC master 8259.
#define		PCL_OCW2	0x20		// PC master 8259.
#define		PCH_OCW1	0xa1		// PC slave 8259.
#define		PCH_OCW2	0xa0		// PC slave 8259.

#define		PCL_INT_BASE	0x08		// Base interrupt number.
#define		PCH_INT_BASE	0x70		// Base interrupt number.
//
// PIC operational command words.
//
#define		EOI_CMD		0x20		// OCW2:Non specific EOI
//
//------------------------------------------------------------------------------
//
// Timer interrupt number.
//
#define		TMR_INT		0x08		// System timer interrupt vector #.
						// (Interrupt 0x1c is called by 0x08)
//
// Timer chip addresses and control bytes.
//
#define		TMR_0		0x40
#define		TMR_CTRL	0x43
#define		TMR_0_LD	0x36
#define		TMR_0_LCH	0x00		// Latches timer 0 count.
//
//------------------------------------------------------------------------------
//
// Register initialization values.
//
//------------------------------------------------------------------------------
//
// Baudrate and bit timing setup.
//
// Tscl = 2 x Tclk x ( BRP + 1 )  where Tclk = 1 / Fclk
// Tscl = 2 x ( 1 / Fclk ) x ( BRP + 1 )
//	-so-
// BRP = ( ( Tscl x Fclk ) / 2 ) - 1
//
// Tbit = 1(Tscl) + TSEG1(Tscl) + TSEG2(Tscl)
//
// ODVA CAN specs : 16 Tscl make up 1 bit time; 1 sync, 13 TSEG1, 2 TSEG2.
//		  : SJW = 0
//		  : One sample per bit.
//
// Example at 125Kbd using 16MHz crystal:
//
//	Tbit = 1 / 125,000 = 8uS = .000008
//	Tscl = Tbit / 16 = 8/16uS = .0000005	(Because 16 Tscl in 1 Tbit)
//		-so-
//	BRP  = ( ( .0000005 x 16E6 ) / 2 ) - 1
//	     = 3
//
//	BTR0 = SJW + BRP
//	     = 0 _ 3
//	     = 0 0 _ 0 0 0 0 1 1
//	     = 0x03
//
//	BTR1 = SAM + TSEG2 + TSEG1
//	     = 0 _ 1 _ 12
//	     = 0 _ 0 0 1 _ 1 1 0 0
//	     = 0x1c
//
// BTR register for 125KHz
//
#define		BTR0_INIT_125	0x03
#define		BTR1_INIT_125	0x1c
//
#define         SJW_125         0
#define         BRP_125         3
#define         SAM_125         0
#define         TS2_125         1
#define         TS1_125         0x0c
//
// BTR register for 250KHz
//
#define		BTR0_INIT_250	0x01
#define		BTR1_INIT_250	0x1c
//
#define         SJW_250         0
#define         BRP_250         1
#define         SAM_250         0
#define         TS2_250         1
#define         TS1_250         0x0c
//
// BTR register for 500KHz
//
#define		BTR0_INIT_500	0x00
#define		BTR1_INIT_500	0x1c
//
#define         SJW_500         0
#define         BRP_500         0
#define         SAM_500         0
#define         TS2_500         1
#define         TS1_500         0x0c
//
//------------------------------------------------------------------------------
//
// Output control register.
//
//    OC1    +     OC0      +   OCMODE
//    111          110            10		= 0xfa
// Inverting  Non-inverting  Normal Output
//   (TX1)        (TX0)
//
#define		OCR_INIT	0xfa
//
//------------------------------------------------------------------------------
//
// Clock divider register.
//
// CAN_MODE + CBP + RXINTEN + '0' + CLOCK_OFF + CLOCK_DIV
//    1        1       0       0        1          000		= 0xc8
//
#define		CDR_INIT	0xc8
//
//------------------------------------------------------------------------------
//
// Handy way to access data within a 32 bit number.
//
typedef union glv
{
	long	l;
	short	i[2];
        char	c[4];
        ulong	ul;
        ushort	ui[2];
	uchar	uc[4];
};
//
//------------------------------------------------------------------------------
//
// Global init status.
short	canInitStatus;
//
//------------------------------------------------------------------------------
//
// Transfers 'canBuffer' data to 'txBuf' in 'canObject'.
//
static void buffer2raw( struct canObject *cp , struct canBuffer *buf )
{
        uchar		far *rdp;
	uchar		c , *dp;
	uchar		infoBits;
	union glv	id;

	// Point to chip data area in raw memory buffer.
        rdp = &cp->txBuf[ cp->txIn ].info;
        // Increment buffer index.
        if ( ++cp->txIn >= cp->txBufSize ) cp->txIn = 0;
	// Zero 'info' byte.
        infoBits = 0;
	// Extended frame?
	if ( buf->type == 1 ) infoBits |= FF_BIT;
	// Remote transmission request?
	if ( buf->rtr == 1 ) infoBits |= RTR_BIT;
        // How many bytes?
        infoBits |= ( buf->cnt & DLC_BITS );
        // Store 'info' byte.
        *rdp++ = infoBits;
        // Get identifier.
        id.l = buf->id;
        // If extended frame.
        if ( buf->type == 1 )
        {
           // Left justify 29 bit id in 32 bit long value.
           id.l <<= 3;
           // Load 4 byte identifier into raw buffer.
	   *rdp++ = id.uc[3];
           *rdp++ = id.uc[2];
           *rdp++ = id.uc[1];
           *rdp++ = id.uc[0];
        }
        else
        {
           // Left justify 11 bit id in 16 bit integer value.
           id.i[0] <<= 5;
           // Load 2 byte identifier into chip.
           *rdp++ = id.uc[1];
           *rdp++ = id.uc[0];
        }
        // Point to data in user's buffer.
        dp = buf->data;
        // Load data.
        for ( c = buf->cnt ; c-- ; ) *rdp++ = *dp++;
}
//
//------------------------------------------------------------------------------
//
// Transfers data from 'rxBuf' in 'canObject' to 'canBuffer'.
//
static void raw2buffer( struct canObject *cp , struct canBuffer *buf )
{
	uchar			far *rdp;
	uchar			c , *dp;
	uchar			infoBits = 0;
	union glv		id;
	struct rawCanBuffer far	*rbuf;

	// Point to chip data in raw memory buffer.
	rbuf = (struct rawCanBuffer far *)&cp->rxBuf[ cp->rxOut ];
	// Increment buffer index.
	if ( ++cp->rxOut >= cp->rxBufSize ) cp->rxOut = 0;
	// Store time stamp.
	buf->time = rbuf->time;
	// Get frame information.
        infoBits = rbuf->info;
	// Extended frame?
        buf->type = ( infoBits & FF_BIT ) ? 1 : 0;
	// Remote transmission request?
        buf->rtr = ( infoBits & RTR_BIT ) ? 1 : 0;
        // How many bytes?
        buf->cnt = ( infoBits & DLC_BITS );
        // Point to raw data.
	rdp = &rbuf->data[0];
        // If extended frame.
        if ( buf->type == 1 )
        {
           // Get 4 byte identifier from raw buffer.
           id.uc[3] = *rdp++;
           id.uc[2] = *rdp++;
           id.uc[1] = *rdp++;
           id.uc[0] = *rdp++;
           // Right justify 29 bit identifier in 32 bit word.
           id.l >>= 3;
        }
        else
        {
           // Get 2 byte identifier from raw buffer; zero high word.
           id.ui[1] = 0;
           id.uc[1] = *rdp++;
           id.uc[0] = *rdp++;
           // Right justify 11 bit identifer in 16 bit word.
           id.ui[0] >>= 5;
        }
        // Set identifier.
        buf->id = id.l;
        // Point to user's data.
        dp = buf->data;
        // Load data.
        for ( c = buf->cnt ; c-- ; ) *dp++ = *rdp++;
}
//
//------------------------------------------------------------------------------
//
// Loads chip with data from raw memory buffer and initiates transmission.
//
// This routine is called from within the ISR (while interrupts are disabled),
// or from within the 'canSend()' service when interrupts aren't used,
// or from within the 'canSend()' service when no transmission is currently
// in process.
// Therefore, we can increment 'txCount' here without worrying about an
// interrupt interfering.
//
static void raw2chip( struct canObject *cp )
{
        uchar far	*s;
	ushort		i , port;

	// Point to start of chip info in raw memory buffer.
	s = &cp->txBuf[ cp->txOut ].info;
	// Point to first tx chip address.
	port = TXB_REG;
	// Transfer data from memory to chip.
        for ( i = 13 ; i-- ; )
        {
           outportb( port++ , *s++ );
        }
        // Increment index.
        if ( ++cp->txOut >= cp->txBufSize ) cp->txOut = 0;
        // Decrement number of packets in buffer.
        --cp->txCount;
        // Enable transmission; do this last so interrupts can't screw
	// with process.
	outportb( CMR_REG , TR_BIT );
}
//
//------------------------------------------------------------------------------
//
// Loads memory buffer with data from chip.
//
// This routine is called from within the ISR (while interrupts are disabled),
// or from within the 'canRecv()' service when interrupts aren't used.
// Therefore, we can increment 'rxcount' here without worrying about an
// interrupt interfering.
//
static void chip2raw( struct canObject *cp )
{
	ushort			i , port;
        uchar far		*s;
	struct rawCanBuffer far	*rbp;

	// Point to current raw memory buffer.
	rbp = (struct rawCanBuffer far *)&cp->rxBuf[ cp->rxIn ];
	// Set to zero time?
	if ( cp->timeTrig )
	{
	   // Load zero point with current time.
	   cp->timeZero = tmrTics;
	   // Reset flag.
	   cp->timeTrig = 0;
	}
	// Get time stamp.
	rbp->time = tmrTics - cp->timeZero;
	// Point to chip data in buffer.
	s = &rbp->info;
	// Point to first rx chip address.
	port = RXB_REG;
	// Transfer data from chip to memory.
        for ( i = 13 ; i-- ; )
        {
           *s++ = inportb( port++ );
        }
        // Increment index.
        if ( ++cp->rxIn >= cp->rxBufSize ) cp->rxIn = 0;
        // Increment number of packets in buffer.
        ++cp->rxCount;
        // Release buffer.
        outportb( CMR_REG , RRB_BIT );
}
//
//-----------------------------------------------------------------------------
//
// Interrupt service; called by ISR.
//
static void intService( struct canObject *cp )
{
	uchar	intReg;

	// Create infinite loop.
        for (;;)
        {

        // Grab interrupt register status.
        // This will reset all flags but the RX int bit.
        // RX int is reset when hardware buffer is released.
	intReg = inportb( IR_REG );
        // Exit loop if no ints.
        if ( intReg == 0 ) break;
	// Loop until all interrupt sources satisfied.
	while ( intReg != 0 )
	{
           //
           // Receiver full?
	   //
	   if ( intReg & RI_BIT )
	   {
	      // If buffer full.
	      if ( cp->rxCount >= cp->rxBufSize )
	      {
		 // Set overrun flag, discard hardware buffer.
		 cp->rxOverrun = TRUE;
	      }
	      else
	      {
		 // Transfer data from chip to memory buffer.
		 chip2raw( cp );
	      }
	      // Reset temporary interrupt bit.
	      intReg &= ~RI_BIT;
	   }
	   //
	   // Transmitter emtpy?
	   //
	   else if ( intReg & TI_BIT )
	   {
	      // If no packets to send.
	      if ( cp->txCount == 0 )
	      {
		 // Clear flag.
		 cp->txRunning = 0;
	      }
	      else
	      {
		 // Transfer data from memory buffer to chip.
		 raw2chip( cp );
	      }
	      // Reset temporary interrupt bit.
	      intReg &= ~TI_BIT;
	   }
	   //
	   // Must have been an error condition.
	   //
	   else
	   {
	      // Increment error counter.
	      ++cp->errorCnt;
	      // Accumulate error bits via ORing.
	      cp->errorFlags |= ( intReg & ~( RI_BIT + TI_BIT ) );
	      // Save error status.
	      cp->errorLog[ cp->errorIdx ] = intReg;
	      // Increment index; wrap around.
	      if ( ++cp->errorIdx >= MAX_ERROR ) cp->errorIdx = 0;
	      // Clear all error flags.
              intReg &= ( RI_BIT + TI_BIT );
           }
	} // while

	} // infinite loop.

	// Disable interrupts for EOI.
        // _DBUG: We haven't enabled them...
	__asm cli
	// EOI for system PIC low.
	outportb( PCL_OCW2 , EOI_CMD );
	if ( cp->irq >= 8 )
	   // EOI for system PIC high.
	   outportb( PCH_OCW2 , EOI_CMD );
	// NOTE: No need to enable interrupts before returning because
	// the flags will be popped off the stack as part of the IRET.
}
//
//------------------------------------------------------------------------------
//
// Pointers to possible 'canObject' structures; one for each interrupt source.
// These are passed to 'intService()' below by the individual isr's.
//
static struct canObject	*cp2_9 , *cp3 , *cp4 , *cp5 , *cp6 , *cp7 ,
			*cp10 , *cp11 , *cp12 , *cp14 , *cp15;

#ifdef	__cplusplus
static void interrupt far irq2_9(...)
#else
static void interrupt far irq2_9()
#endif
{
	intService( cp2_9 );
}

#ifdef	__cplusplus
static void interrupt far irq3(...)
#else
static void interrupt far irq3()
#endif
{
	intService( cp3 );
}

#ifdef	__cplusplus
static void interrupt far irq4(...)
#else
static void interrupt far irq4()
#endif
{
	intService( cp4 );
}

#ifdef	__cplusplus
static void interrupt far irq5(...)
#else
static void interrupt far irq5()
#endif
{
	intService( cp5 );
}

#ifdef	__cplusplus
static void interrupt far irq6(...)
#else
static void interrupt far irq6()
#endif
{
	intService( cp6 );
}

#ifdef	__cplusplus
static void interrupt far irq7(...)
#else
static void interrupt far irq7()
#endif
{
	intService( cp7 );
}

#ifdef	__cplusplus
static void interrupt far irq10(...)
#else
static void interrupt far irq10()
#endif
{
	intService( cp10 );
}

#ifdef	__cplusplus
static void interrupt far irq11(...)
#else
static void interrupt far irq11()
#endif
{
	intService( cp11 );
}

#ifdef	__cplusplus
static void interrupt far irq12(...)
#else
static void interrupt far irq12()
#endif
{
	intService( cp12 );
}

#ifdef	__cplusplus
static void interrupt far irq14(...)
#else
static void interrupt far irq14()
#endif
{
	intService( cp14 );
}

#ifdef	__cplusplus
static void interrupt far irq15(...)
#else
static void interrupt far irq15()
#endif
{
	intService( cp15 );
}
//
//------------------------------------------------------------------------------
//
// Here to hook PC interrupt.
//
static short hookIrq( struct canObject *cp , short irqNumber )
{
	uchar		pcIrqMask;
	#ifdef	__cplusplus
	void (interrupt	far *irqPtr)(...);
	#else
	void (interrupt far *irqPtr)();
	#endif

	// Check to see if it a valid IRQ.
	// Set IRQ's mask & pointer to new isr.
	// Return FAILURE if unsupported IRQ.
	switch ( irqNumber )
	{
        case 2:
        case 9:  pcIrqMask = 0x02; irqPtr = irq2_9 ; cp2_9 = cp; break;
	case 3:  pcIrqMask = 0x08; irqPtr = irq3  ; cp3  = cp ; break;
	case 4:  pcIrqMask = 0x10; irqPtr = irq4  ; cp4  = cp ; break;
	case 5:  pcIrqMask = 0x20; irqPtr = irq5  ; cp5  = cp ; break;
	case 6:  pcIrqMask = 0x40; irqPtr = irq6  ; cp6  = cp ; break;
	case 7:  pcIrqMask = 0x80; irqPtr = irq7  ; cp7  = cp ; break;
	case 10: pcIrqMask = 0x04; irqPtr = irq10 ; cp10 = cp ; break;
	case 11: pcIrqMask = 0x08; irqPtr = irq11 ; cp11 = cp ; break;
	case 12: pcIrqMask = 0x10; irqPtr = irq12 ; cp12 = cp ; break;
	case 14: pcIrqMask = 0x40; irqPtr = irq14 ; cp14 = cp ; break;
	case 15: pcIrqMask = 0x80; irqPtr = irq15 ; cp15 = cp ; break;
	default:
	   return FAILURE;
	}
	// If PC master 8259 (ints 0-7).
	if ( cp->irq < 8 )
	{
	   //First we get the original PC's 8259 interrupt mask
	   cp->oldImrMask = inportb( PCL_OCW1 );
	   // Then save the old interrupt vector.
	   cp->oldHandler = getvect( cp->irq + PCL_INT_BASE );
	   // Now set the new vector.
	   setvect( cp->irq + PCL_INT_BASE , irqPtr );
	   // Now unmask the interrupts.
	   outportb( PCL_OCW1 , ~pcIrqMask & cp->oldImrMask );
	}
	// Else slave 8259 (ints 8-15).
	else
	{
	   //First we get the original PC's 8259 interrupt mask
	   cp->oldImrMask = inportb( PCH_OCW1 );
	   // Then save the old interrupt vector.
	   cp->oldHandler = getvect( cp->irq - 8 + PCH_INT_BASE );
	   // Now set the new vector.
	   setvect( cp->irq - 8 + PCH_INT_BASE , irqPtr );
	   // Now unmask the interrupts.
	   outportb( PCH_OCW1 , ~pcIrqMask & cp->oldImrMask );
	}
	// Return success.
	return SUCCESS;
}
//
//------------------------------------------------------------------------------
//
// Here to unhook PC interrupt.
//
static void unhookIrq( struct canObject *cp )
{
	// Return if irq not installed.
	if ( cp->irq == -1 ) return;
	// Disable interrupts.
	__asm pushf
	__asm cli
	// PC Master?
	if ( cp->irq < 8 )
	{
	   // Put old mask back.
	   outportb( PCL_OCW1 , cp->oldImrMask );
	   // Put the old vectro back.
	   setvect( cp->irq + PCL_INT_BASE , cp->oldHandler );
	}
	// PC Slave.
	else
	{
	   // Put old mask back.
	   outportb( PCH_OCW1 , cp->oldImrMask );
	   // Put the old vectro back.
	   setvect( cp->irq - 8 + PCH_INT_BASE , cp->oldHandler );
	}
	// Clear the current irq number.
	cp->irq = -1;
	// Put interrupts back where they were.
	__asm popf
}
//
//------------------------------------------------------------------------------
//
//	CAN Object functions.
//
//------------------------------------------------------------------------------
//
//
// Puts CAN chip in reset mode.
//
short canReset( struct canObject *cp )
{
	short	loopCount = 1000;	// _DBUG: Use timer?

        // Check pointer.
        if ( cp == 0 ) return FAILURE;
	// Set reset bit.
        cp->modeReg |= RM_BIT;
        // While we can still count.
        while ( loopCount )
        {
           // Set reset bit in mode register.
	   outportb( MOD_REG , cp->modeReg );
           // Exit loop if reset.
	   if ( ( inportb( MOD_REG ) & RM_BIT ) == RM_BIT ) break;
           // Decrement loop counter.
           --loopCount;
        }
        // Failure if count is 0, success otherwise.
	if ( loopCount == 0 )
	   return FAILURE;
	else
	   return SUCCESS;
}
//
//------------------------------------------------------------------------------
//
// Takes CAN chip out of reset mode.
//
short canEnable( struct canObject *cp )
{
	short	loopCount = 1000;	// _DBUG: Use timer?

        // Check pointer.
        if ( cp == 0 ) return FAILURE;
        // Clear reset bit.
        cp->modeReg &= ~RM_BIT;
        // While we can still count.
	while ( loopCount )
        {
           // Clear reset bit in mode register.
	   outportb( MOD_REG , cp->modeReg );
           // Exit loop if not reset.
           if ( ( inportb( MOD_REG ) & RM_BIT ) == 0 ) break;
           // Decrement loop counter.
           --loopCount;
        }
        // Failure if count is 0, success otherwise.
	if ( loopCount == 0 )
           return FAILURE;
        else
           return SUCCESS;
}
//
// ----------------------------------------------------------------------------
//
// Sets or clears listen only mode.
//
short canListenOnly( struct canObject *cp , short onOff )
{
        char    wasEnabled;

        // Check pointer.
        if ( cp == 0 ) return FAILURE;
        // Are we in reset?
        if ( inportb( MOD_REG ) & RM_BIT )
        {
           // No need to reset chip when done.
           wasEnabled = 0;
	}
        else
        {
           // Show we were running.
           wasEnabled = 1;
           // Put the chip in reset.
           canReset( cp );
        }
        // Turn listen only on or off?.
        if ( onOff == TRUE )
           // Turn it on.
           cp->modeReg |= LOM_BIT;
        else
           // Turn it off.
           cp->modeReg &= ~LOM_BIT;
	// Should we take the chip out of reset?
        if ( wasEnabled )
           // Enable chip.
	   canEnable( cp );
	// And return.
	return SUCCESS;
}
//
//------------------------------------------------------------------------------
//
// Sets baud rate by explicitly specifying bit timing registers.
//
short canSetBtr( struct canObject *cp , short sjw ,
                                        short prescale,
                                        short tseg1,
                                        short tseg2,
                                        short sampling )
{
        short   i;
	char	wasEnabled;

        // Check pointer.
        if ( cp == 0 ) return FAILURE;
        // Are we in reset?
        if ( inportb( MOD_REG ) & RM_BIT )
        {
           // No need to reset chip when done.
           wasEnabled = 0;
	}
        else
        {
           // Show we were running.
           wasEnabled = 1;
           // Put the chip in reset.
           canReset( cp );
        }
	// Set BTR0.
        outportb( BTR0_REG , ( sjw << 6 ) + ( prescale & 0x3f ) );
        // Set BTR1.
        outportb( BTR1_REG , ( sampling << 7 ) + ( ( tseg2 & 7 ) << 4 ) + ( tseg1 & 0x0f ) );
	// Should we take the chip out of reset?
        if ( wasEnabled )
           // Enable chip.
	   canEnable( cp );
	// And return.
	return SUCCESS;
}
//
//------------------------------------------------------------------------------
//
// Here to set baud rate for can object.
// This will assume that the crystal is 16MHz.
//
short canSetBaud( struct canObject *cp , short baudrate )
{
	char	wasEnabled;

        // Check pointer.
        if ( cp == 0 ) return FAILURE;
        // Are we in reset?
        if ( inportb( MOD_REG ) & RM_BIT )
        {
           // No need to reset chip when done.
           wasEnabled = 0;
	}
        else
        {
           // Show we were running.
           wasEnabled = 1;
           // Put the chip in reset.
           canReset( cp );
        }
	// How fast?
	switch ( baudrate )
	{
        case BAUD_125KB:
	   canSetBtr( cp , SJW_125 , BRP_125 , TS1_125 , TS2_125 , SAM_125 );
	   break;
	case BAUD_250KB:
	   canSetBtr( cp , SJW_250 , BRP_250 , TS1_250 , TS2_250 , SAM_250 );
	   break;
	case BAUD_500KB:
	   canSetBtr( cp , SJW_500 , BRP_500 , TS1_500 , TS2_500 , SAM_500 );
           break;
        default:
           return FAILURE;
        }
	// Should we take the chip out of reset?
        if ( wasEnabled )
           // Enable chip.
	   canEnable( cp );
	// And return.
	return SUCCESS;
}
//
//------------------------------------------------------------------------------
//
// Here to set the acceptance filter for can object.
//
short canSetFilter( struct canObject *cp , struct canFilter *filter )
{
	uchar		wasEnabled;
	uchar		acr[4];
	uchar		amr[4];
	short		retval = SUCCESS;
	union glv	gid;

	// Check pointer.
	if ( cp == 0 ) return FAILURE;
	// Set all codes & masks to '0'.
	acr[0] = acr[1] = acr[2] = acr[3] = 0;
	amr[0] = amr[1] = amr[2] = amr[3] = 0;
        // Are we in reset?
        if ( inportb( MOD_REG ) & RM_BIT )
        {
           // No need to reset chip when done.
	   wasEnabled = 0;
	}
        else
        {
           // Show we were running.
           wasEnabled = 1;
           // Put the chip in reset.
	   canReset( cp );
        }
        // Which type of filter.
        switch ( filter->type )
        {
        case STD_SNGL:
	   //
           // Acceptance code registers.
           //
           // Slip into something more comfortable.
           gid.l = filter->idCode1;
           // Left justify 11 bit code in 16 bit integer.
	   gid.i[0] <<= 5;
	   // First code register.
	   acr[0] = gid.uc[1];
           // Construct second code register.
	   acr[1] = gid.uc[0] & 0xe0;
           // Set rtr bit?
           if ( filter->rtrCode1 )
	      acr[1] |= 0x10;
	   // Data code registers.
	   acr[2] = filter->dataCode[0];
	   acr[3] = filter->dataCode[1];
           //
           // Acceptance mask registers.
           //
	   // Slip into something more comfortable.
           gid.l = filter->idMask1;
           // Left justify 11 bit code in 16 bit integer.
	   gid.i[0] <<= 5;
	   // First mask register.
	   amr[0] = gid.uc[1];
	   // Construct second mask register.
	   amr[1] = gid.uc[0] & 0xe0;
	   // Set rtr bit?
           if ( filter->rtrMask1 )
	      amr[1] |= 0x10;
           // Set unused mask bits to '1'.
	   amr[1] |= 0x0f;
           // Data mask registers.
	   amr[2] = filter->dataMask[0];
	   amr[3] = filter->dataMask[1];
	   // Set dual filter mode bit.
	   cp->modeReg |= AFM_BIT;
           break;
        case STD_DUAL:
	   //
           //------------------------------------
           // Filter 1 acceptance code registers.
           //------------------------------------
           //
           // Slip into something more comfortable.
	   gid.l = filter->idCode1;
	   // Left justify 11 bit code in 16 bit integer.
	   gid.i[0] <<= 5;
	   // First id code value.
	   acr[0] = gid.uc[1];
           // Construct second id code value.
           // Mask off all but top 3 bits.
	   acr[1] |= gid.uc[0] & 0xe0;
           // Set rtr bit?
           if ( filter->rtrCode1 )
	      acr[1] |= 0x10;
	   // Top 4 bits of data code included.
	   acr[1] |= filter->dataCode[0] >> 4;
           // Bottom 4 bits of data code elsewhere.
	   acr[3] |= filter->dataCode[0] & 0x0f;
	   //
           // Acceptance mask registers, first filter.
           //
           // Slip into something more comfortable.
           gid.l = filter->idMask1;
	   // Left justify 11 bit code in 16 bit integer.
	   gid.i[0] <<= 5;
	   // First id mask value.
	   amr[0] = gid.uc[1];
	   // Construct second id mask value.
	   amr[1] = gid.uc[0] & 0xe0;
           // Set rtr bit?
           if ( filter->rtrCode1 )
	      amr[1] |= 0x10;
           // Top 4 bits of data mask included.
	   amr[1] |= filter->dataMask[0] >> 4;
	   // Bottom 4 bits of data mask elsewhere.
	   amr[3] |= filter->dataMask[0] & 0x0f;
           //
	   //------------------------------------
           // Filter 2 acceptance code registers.
           //------------------------------------
           //
           // Slip into something more comfortable.
           gid.l = filter->idCode2;
           // Left justify 11 bit code in 16 bit integer.
	   gid.i[0] <<= 5;
	   // First id code value.
	   acr[2] = gid.uc[1];
	   // Construct second id code value.
	   acr[3] |= gid.uc[0] & 0xe0;
           // Set rtr bit?
           if ( filter->rtrCode2 )
	      acr[3] |= 0x10;
	   //
           // Acceptance mask registers, second filter.
	   //
           // Slip into something more comfortable.
           gid.l = filter->idMask2;
	   // Left justify 11 bit code in 16 bit integer.
	   gid.i[0] <<= 5;
	   // First id mask value.
	   amr[2] = gid.uc[1];
	   // Construct second id mask value.
	   amr[3] |= gid.uc[0] & 0xe0;
	   // Set rtr bit?
	   if ( filter->rtrCode2 )
	      amr[3] |= 0x10;
	   // Clear dual filter mode bit.
	   cp->modeReg &= ~AFM_BIT;
	   break;
	case EXT_SNGL:
	   //
	   // Acceptance code registers.
	   //
	   // Slip into something more comfortable.
	   gid.l = filter->idCode1;
	   // Left justify 29 bit code in 32 bit integer.
	   gid.l <<= 3;
	   // Code registers.
	   acr[0] = gid.uc[3];
	   acr[1] = gid.uc[2];
	   acr[2] = gid.uc[1];
	   acr[3] = gid.uc[0];
           // Set rtr bit?
           if ( filter->rtrCode1 )
	      acr[3] |= 0x04;
	   //
           // Acceptance mask registers.
           //
           // Slip into something more comfortable.
           gid.l = filter->idMask1;
	   // Left justify 29 bit code in 32 bit integer.
	   gid.l <<= 3;
	   // Code registers.
	   amr[0] = gid.uc[3];
	   amr[1] = gid.uc[2];
	   amr[2] = gid.uc[1];
	   amr[3] = gid.uc[0];
	   // Set rtr bit?
	   if ( filter->rtrMask1 )
	      amr[3] |= 0x04;
	   // Set unused mask bits to '1'.
	   amr[3] |= 0x03;
	   // Set dual filter mode bit.
	   cp->modeReg |= AFM_BIT;
	   break;
	case EXT_DUAL:
	   //
	   //------------------------------------
	   // Filter 1 acceptance code registers.
	   //------------------------------------
	   //
	   // Slip into something more comfortable.
	   gid.l = filter->idCode1;
	   // Code registers.
	   acr[0] = gid.uc[1];
	   acr[1] = gid.uc[0];
	   //
	   // Acceptance mask registers.
	   //
	   // Slip into something more comfortable.
	   gid.l = filter->idMask1;
	   // Code registers.
	   amr[0] = gid.uc[1];
	   amr[1] = gid.uc[0];
	   //
	   //------------------------------------
	   // Filter 2 acceptance code registers.
	   //------------------------------------
	   //
	   // Slip into something more comfortable.
	   gid.l = filter->idCode2;
	   // Code registers.
	   acr[2] = gid.uc[1];
	   acr[3] = gid.uc[0];
	   //
	   // Acceptance mask registers.
	   //
	   // Slip into something more comfortable.
	   gid.l = filter->idMask2;
	   // Code registers.
	   amr[2] = gid.uc[1];
	   amr[3] = gid.uc[0];
           // Set dual filter mode bit.
	   cp->modeReg &= ~AFM_BIT;
           break;
        default:
	   retval = FAILURE;
	}
        // Now output the registers.
	outportb( ACR0_REG , acr[0] );
	outportb( ACR1_REG , acr[1] );
	outportb( ACR2_REG , acr[2] );
	outportb( ACR3_REG , acr[3] );
	//
	outportb( AMR0_REG , amr[0] );
	outportb( AMR1_REG , amr[1] );
	outportb( AMR2_REG , amr[2] );
	outportb( AMR3_REG , amr[3] );
        // Set mode register; AFM bit can only be changed while in reset.
        outportb( MOD_REG , cp->modeReg );
        // Should we take the chip out of reset?
        if ( wasEnabled )
           // Enable chip.
           canEnable( cp );
	// And return.
	return retval;
}
//
//------------------------------------------------------------------------------
//
// Here to send a packet.
// Returns SUCCESS or FAILURE.
//
short canSend( struct canObject *cp , struct canBuffer *buf )
{
        // Check pointer.
        if ( cp == 0 ) return FAILURE;
        // If buffer full.
        if ( cp->txCount == cp->txBufSize )
        {
           // Exit failure.
           return FAILURE;
        }
        // Add packet to buffer.
        buffer2raw( cp , buf );
	// Are we using interrupts?
	if ( cp->irqUsed )
        {
           // Disable interrupts.
	   __asm pushf
           __asm cli
           // Increment count when interrupts are disabled.
           ++cp->txCount;
           // If we're already transmitting.
           if ( cp->txRunning )
           {
	      // Restore interrupts.
              __asm popf
           }
           // Else first one in buffer.
           else
           {
              // Restore interrupts. Since we aren't running, we don't have
              // to worry that our TX interrupt will occur during buffer
              // manipulation.
              __asm popf
              // _DBUG: Possible to hang here?
              // Wait for hardware transmit buffer to empty.
              while ( ( inportb( SR_REG ) & TBS_BIT ) == 0 );
	      // Set flag before you send so an interrupt won't be able
              // to screw up the process.
              cp->txRunning = 1;
              // Send it.
              raw2chip( cp );
           }
        }
	// Else no interrupts.
        else
        {
           // _DBUG: Possible to hang here?
           // Wait for hardware transmit buffer to empty.
           while ( ( inportb( SR_REG ) & TBS_BIT ) == 0 );
           // Send it.
           raw2chip( cp );
        }
        // And return.
	return SUCCESS;
}
//
//------------------------------------------------------------------------------
//
// Here to receive a packet.
//
short canRecv( struct canObject *cp , struct canBuffer *buf )
{
        // Check pointer.
        if ( cp == 0 ) return FAILURE;
	// Are we using interrupts?
	if ( cp->irqUsed )
        {
           // Just return if no packets.
           if ( cp->rxCount == 0 ) return FAILURE;
           // Convert memory buffer & transfer.
           raw2buffer( cp , buf );
           // Disable interrupts.
           __asm pushf
           __asm cli
           // Decrement count.
           --cp->rxCount;
           // Restore interrupts.
           __asm popf
	}
        else
        {
	   // Just return if no packet in chip.
	   if ( ( inportb( SR_REG ) & RBS_BIT ) == 0 ) return FAILURE;
	   // Transfer data from chip to memory buffer.
           chip2raw( cp );
	   // Convert memory buffer & transfer.
           raw2buffer( cp , buf );
           // Decrement count.
           --cp->rxCount;
        }
        // And return.
        return SUCCESS;
}
//
//------------------------------------------------------------------------------
//
// Here to return status.
//
ushort canStatus( struct canObject *cp )
{
	ushort	retval = 0;
	uchar	statReg;

        // Check pointer.
        if ( cp == 0 ) return 0;
	// Get status register.
        statReg = inportb( SR_REG );
	// Overrun?
	if ( statReg & DOS_BIT ) retval |= STAT_OVERRUN;
        // Bus error?
	if ( statReg & ES_BIT  ) retval |= STAT_ERROR;
        // Bus off?
	if ( statReg & BS_BIT  ) retval |= STAT_BUSOFF;
	// Are we using interrupts?
        if ( cp->irqUsed == TRUE )
	{
	   // Are there any packets in receive buffer?
	   if ( cp->rxCount != 0 ) retval |= STAT_RXRDY;
	   // Is transmit buffer full?
	   if ( cp->txCount < cp->txBufSize ) retval |= STAT_TXRDY;
	   // Is tx buffer empty?
           if ( cp->txCount == 0 )
	   {
              // Is hardware buffer empty and transmission complete?
              if ( ( statReg & TBS_BIT ) && ( statReg & TCS_BIT ) )
                 // Then we're completely done.
		 retval |= STAT_TXCOMPLETE;
	   }
        }
        else
        {
           // Are there any packets in receive buffer?
	   if ( statReg & RBS_BIT ) retval |= STAT_RXRDY;
           // Can you load a new packet into hardware transmit buffer?
	   if ( statReg & TBS_BIT ) retval |= STAT_TXRDY;
	   // Is hardware buffer empty and transmission complete?
           if ( ( statReg & TBS_BIT ) && ( statReg & TCS_BIT ) )
              // Then we're completely done.
	      retval |= STAT_TXCOMPLETE;
        }
	return retval;
}
//
//------------------------------------------------------------------------------
//
// Here to return accumulated error bits.
//
ushort canError( struct canObject *cp )
{
	ushort		retval;

        // Check pointer.
        if ( cp == 0 ) return 0;
	// Disable interrupts.
	__asm	pushf
	__asm	cli
	// Get error status.
	retval = cp->errorFlags;
	// Clear flags.
	cp->errorFlags = 0;
	// Restore interrupts.
	__asm	popf
	// And return.
	return retval;
}
//
//------------------------------------------------------------------------------
//
// Here to purge the tx buffer.
//
void canTxPurge( struct canObject *cp )
{
        // Check pointer.
        if ( cp == 0 ) return;
	// Disable interrupts.
        __asm	pushf
        __asm	cli
	// Reset the transmit section.
	cp->txRunning = 0;
	cp->txCount = 0;
	cp->txIn = 0;
        cp->txOut = 0;
        // Restore interrupts.
        __asm	popf
}
//
//------------------------------------------------------------------------------
//
// Here to purge the rx buffer.
//
void canRxPurge( struct canObject *cp )
{
        // Check pointer.
        if ( cp == 0 ) return;
	// Disable interrupts.
        __asm	pushf
        __asm	cli
	// Reset the transmit section.
	cp->rxCount = 0;
	cp->rxIn = 0;
        cp->rxOut = 0;
        // Restore interrupts.
        __asm	popf
}
//
//------------------------------------------------------------------------------
//
// Here to recover from error or bus-off.
//
void canRecover( struct canObject *cp )
{
	uchar		ieReg;

        // Check pointer.
        if ( cp == 0 ) return;
	// Capture the interrupt enable register.
	ieReg = inportb( IER_REG );
	// Disable all interrupts.
	outportb( IER_REG , 0 );
	// Reset the chip.
	canReset( cp );
        // Reset rx error register.
        outportb( RXERR_REG , 0 );
        // Purge any packets waiting to be read.
        canRxPurge( cp );
        // Purge any packets waiting to be sent.
        canTxPurge( cp );
	// Give a transmit abort for good measure.
	outportb( CMR_REG , AT_BIT );
	// Read interrupt register (to clear int?).
	inportb( IR_REG );
	// Enable the chip.
	canEnable( cp );
	// Restore the interrupt enable register.
	outportb( IER_REG , ieReg );
}
//
//------------------------------------------------------------------------------
//
// Timer setup/control fucntion.
//
void canTime( struct canObject *cp , short fcn )
{
        // Check pointer.
        if ( cp == 0 ) return;
	// Which service?
        switch ( fcn )
        {
        //
	// Rest time base.
	//
        case TIME_RESET:
	   // Reset base for time stamps.
	   cp->timeZero = tmrTics;
           break;
        //
	// Set trigger so next received packet gets time stamp of '0'.
        //
        case TIME_TRIG:
           // Set flag.
           cp->timeTrig = 1;
           break;
        }
}
//
//------------------------------------------------------------------------------
//
// Here to create and initialize a new can object.
// Returns a pointer to the new object.
// This function is passed explicit values for the bit timing registers.
//
struct canObject *canInitBtr( short irqNum ,
			   ushort ioAddress ,
			   short sjw ,
                           short prescale ,
                           short tseg1 ,
                           short tseg2 ,
                           short sampling ,
			   struct canFilter *filter ,
			   short txSize ,
			   short rxSize )
{
	ushort			i;
	struct canObject	*cp;
	struct timer		*tmr;

	// Assume success.
	canInitStatus = SUCCESS;
	// Initialize timer.
	tmr = tmrInit();
	// Start timer.
	tmrStart( tmr , 18 );
	// Failure if you can't reset the chip.
	for ( ;; )
	{
	   // Set reset bit in mode/command register.
	   outportb( ioAddress , RM_BIT );
	   // Check it.
	   if ( ( inportb( ioAddress ) & RM_BIT ) == RM_BIT ) break;
	   // Exit if timer times out.
	   if ( tmrFlag( tmr ) ) goto NoChipExit;
	}
	// Insure chip is in basic can mode; set clock divider register
	// to reset value.
	outportb( ioAddress + 31 , 0 );
	// Now check that acceptance mask is r/w.
	outportb( ioAddress + 5 , 0x55 );
	if ( inportb( ioAddress + 5 ) != 0x55 ) goto NoChipExit;
	outportb( ioAddress + 5 , 0xaa );
	if ( inportb( ioAddress + 5 ) != 0xaa ) goto NoChipExit;
	// Put chip into pelican mode.
	outportb( ioAddress + 31 , 0x80 );
	// Now check that the (now reserved) register is 0.
	if ( inportb( ioAddress + 5 ) != 0 ) goto NoChipExit;
	// Put chip back in basic can mode.
	outportb( ioAddress + 31 , 0 );
	// Check IRQ number.
	switch ( irqNum )
	{
	case 0:
	case 2:
	case 9:
	case 3:
	case 4:
	case 5:
	case 6:
	case 7:
	case 10:
	case 11:
	case 12:
	case 14:
	case 15:
	   break;
	default:
	   canInitStatus = INIT_BADIRQ;
	   goto ExitHere;
	}
	// Allocate memory for 'canObject'.
	if ( ( cp = malloc( sizeof( struct canObject ) ) ) == NULL )
	{
	   canInitStatus = INIT_MALLOC_CP;
	   goto ExitHere;
	}
	// Allocate memory for tx buffer.
//	if ( ( cp->txBuf = malloc( sizeof( struct rawCanBuffer ) * txSize ) ) == NULL )
	if ( ( cp->txBuf = (struct rawCanBuffer far *)farcalloc( txSize , sizeof( struct rawCanBuffer ) ) ) == NULL )
	{
	   // Release previously allocated storage.
	   free( cp );
	   // And return.
	   canInitStatus = INIT_MALLOC_TX;
	   goto ExitHere;
	}
	// Allocate memory for rx buffer.
//	if ( ( cp->rxBuf = malloc( sizeof( struct rawCanBuffer ) * rxSize ) ) == NULL )
	if ( ( cp->rxBuf = (struct rawCanBuffer far *)farcalloc( rxSize , sizeof( struct rawCanBuffer ) ) ) == NULL )
	{
	   // Release previously allocated storage.
	   farfree( cp->txBuf );
	   free( cp );
	   // And return.
	   canInitStatus = INIT_MALLOC_RX;
	   goto ExitHere;
	}
	// Initialize structure variables.
	cp->inService = 1;
	cp->port = ioAddress;
	cp->modeReg = RM_BIT;		// Reset.
	//
	cp->txRunning = 0;
	cp->errorFlags = 0;
	cp->errorCnt = 0;
	cp->errorIdx = 0;
	for ( i = 0 ; i < MAX_ERROR ; ++i )
	   cp->errorLog[i] = 0;
	cp->rxOverrun = 0;
	cp->timeTrig = 0;
	cp->timeZero = 0;
	//
	cp->rxBufSize = rxSize;
	cp->rxCount = 0;
	cp->rxIn = 0;
	cp->rxOut = 0;
	//
	cp->txBufSize = txSize;
	cp->txCount = 0;
	cp->txIn = 0;
	cp->txOut = 0;
	// Initialize clock divider register; put chip in PeliCan mode.
	outportb( CDR_REG , CDR_INIT );
	// Disable all CAN interrupts.
	outportb( IER_REG , 0 );
	// Define acceptance code and masks.
	canSetFilter( cp , filter );
	// Configure initial baudrate.
	canSetBtr( cp , sjw , prescale , tseg1 , tseg2 , sampling );
	// Initialize output control register.
	outportb( OCR_REG , OCR_INIT );
	// Enable CAN chip.
	canEnable( cp );
	// If interrupts not used.
	if ( irqNum == 0 )
	{
	   // Setup for no interrupts.
	   cp->irqUsed = 0;
	   cp->irq = 0;
	   cp->oldImrMask = 0;
	   cp->oldHandler = 0;
	}
	// Else initialize interrupts.
	else
	{
	   // We are using ints.
	   cp->irqUsed = 1;
	   // Set irq number.
	   cp->irq = (char)irqNum;
	   // Hook interrupt.
	   hookIrq( cp , irqNum );
	   // Enable interrupts on CAN chip.
	   outportb( IER_REG , RI_BIT |
			       TI_BIT |
			       EI_BIT |
			       DOI_BIT |
			       EPI_BIT |
			       BEI_BIT );
	}
ExitHere:
	// Release timer.
	tmrDestroy( tmr );
	// If status is ok.
	if ( canInitStatus == SUCCESS )
	{
	   // Return with pointer to initialized object.
	   return cp;
	}
	else
	   // Return with null pointer.
	   return NULL;
	//
	// Here if no chip detected.
	//
NoChipExit:
	canInitStatus = INIT_NOCHIP;
	goto ExitHere;
}
//
//------------------------------------------------------------------------------
//
// Here to create and initialize a new can object.
// Returns a pointer to the new object.
//
struct canObject *canInit( short irqNum ,
			   ushort ioAddress ,
			   short baudrate ,
			   struct canFilter *filter ,
			   short txSize ,
			   short rxSize )
{
        struct canObject        *cp;

	// How fast?
	switch ( baudrate )
	{
        case BAUD_125KB:
	   cp = canInitBtr( irqNum , ioAddress , SJW_125 , BRP_125 , TS1_125 , TS2_125 , SAM_125 , filter , txSize , rxSize );
	   break;
	case BAUD_250KB:
	   cp = canInitBtr( irqNum , ioAddress , SJW_250 , BRP_250 , TS1_250 , TS2_250 , SAM_250 , filter , txSize , rxSize );
	   break;
	case BAUD_500KB:
	   cp = canInitBtr( irqNum , ioAddress , SJW_500 , BRP_500 , TS1_500 , TS2_500 , SAM_500 , filter , txSize , rxSize );
           break;
	case BAUD_1000KB:
	   cp = canInitBtr( irqNum , ioAddress , 0 , 0 , 4 , 1 , 0 , filter , txSize , rxSize );
           break;
        default:
	   return NULL;
        }
        // Return with pointer.
        return cp;
}
//
//------------------------------------------------------------------------------
//
// Here to disable and delete a can object.
//
void canDestroy( struct canObject *cp )
{
	// Check pointer.
	if ( cp == 0 ) return;
	// Put CAN chip in reset.
	canReset( cp );
	// Unhook interrupt.
	unhookIrq( cp );
	// Release storage.
	farfree( cp->txBuf );
	farfree( cp->rxBuf );
	free( cp );
}
