//-----------------------------------------------------------------------------
//
//	Copyright 2000 Contemporary Controls
//
//-----------------------------------------------------------------------------

#include	<ctype.h>
#include	<dos.h>
#include	<conio.h>
#include	<process.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>

#include	"basics.h"
#include	"can_driv.h"
#include	"ibmkeys.h"
#include	"pctimer.h"
#include	"screen.hpp"
#include	"scrn_fcn.h"

// Set this to just transfer tx message box to rx list box without
// using can boards.
//
#define         LOOPBACK_MODE   0

// Revision history.
//
char	revisionStr[] = "Revision 1.02";
char	revisionDate[] = "November 7,2000";

/*
1.01    5/22/00         First released version.

1.02	11/7/00		Added 1MBit baudrate '4'.
*/

// ----------------------------------------------------------------------------
//
//      Global data.
//
// ----------------------------------------------------------------------------
//
// Group 3 messages are used for communications. We use one of the 5 available
// message ID's ('100').
// This number is the top 5 bits of the identifier; the bottom 6 bits will be
// the MAC ID of the SOURCE node. The MAC ID of the DESTINATION node is encoded
// as the first data byte of the packet.
//
#define         GROUP3_MSG_ID           0x001c          // '11100b'
//
//
// CAN board(s).
//
// These pointers will be the same if one board is attached. This will be the case
// when two computers are used; each one has a board.
//
// These pointers will be different if two boards are used on one computer. This
// will be the case for software test, or if a single cpu demo system is used.
// If two boards are used, the configuration information for the second board will
// be supplied on the command line:
//      cantalk rxIoAddress rxIrqNumber
//
// The second board will be used as 'rxCan'.
// Baudrate will be taken from displayed screen data.
// TX MAC ID will be used as the RX MAC ID for the second board (so every tx
// message should be received and displayed).
//
canObject       *txCan = 0;
canObject       *rxCan = 0;
//
// Initialization flag; set when can board(s) initialized, reset on startup
// and if any info changed.
//
int             canBoardInit = FALSE;
//
// Screen devices.
//
editBox		ebIoAddress;	// CAN board base I/O address.
editBox		ebIrqNumber;	// CAN board IRQ.
editBox         ebBaudrate;     // CAN network baudrate.
editBox		ebTxMacId;	// Destination MAC ID in TX packets.
editBox		ebRxMacId;	// MAC ID assigned to this node.
editBox		ebTxMessage;	// Where user types in TX messages.
listBox         lbRxMessage;    // Where received messages appear.
editBox         ebErrorMsg;     // Error message display.
//
// Screen class instance.
// This provides a linked list to all the 'screen elements'; edit
// boxes and list boxes.
//
screen		scrn1;
//
// Can board operating parameters.
//
unsigned int    ioAddress;
int             irqNumber;
int             baudrate;
int             rxMacId;
int             txMacId;
//
// Command line parameters for second board.
// If 'ioAddress2' is '0', there is no second board.
//
int             ioAddress2 = 0;
int             irqNumber2;
//
// Timer used for keyboard input with redirection.
//
timer           *escTimer;
//
// ----------------------------------------------------------------------------
//
//      Functions to initialize and send messages on the CAN bus.
//
// ----------------------------------------------------------------------------
//
// Here to initialize the can board(s) based on screen data, and upon command
// line data if two boards are present.
//
void initializeCanBoards( void )
{
	char    *s;
        struct
        {
        unsigned int    macId:6;
        unsigned int    group3bits:5;
        unsigned int    pad:5;
        } idField;
        canFilter       filter;

        // Reset init flag.
        canBoardInit = FALSE;
        // Release boards.
        if ( rxCan == txCan )
           canDestroy( txCan );
        else
        {
           canDestroy( rxCan );
           canDestroy( txCan );
        }
        // NULL can board pointers.
        rxCan = txCan = NULL;
        //
        // Convert edit box strings to integers.
        //
        sscanf( ebIoAddress.buf , "%x" , &ioAddress );  // HEX!
        sscanf( ebIrqNumber.buf, "%i" , &irqNumber );
        sscanf( ebBaudrate.buf , "%i" , &baudrate );
        sscanf( ebRxMacId.buf , "%i" , &rxMacId );
        sscanf( ebTxMacId.buf , "%i" , &txMacId );
        //
        // Generate rx filter.
        //
        // Standard frame; single identifier.
        filter.type = STD_SNGL;
        // ID mask; top 5 bits must match; bottom 6 bits don't matter.
        // ***With can filters, '1' bits are don't care***
        filter.idMask1 = 0x003f;
        // ID code; all cantalk nodes will accept these messages.
        filter.idCode1 = GROUP3_MSG_ID << 6;
        // No rtr messages.
        filter.rtrMask1 = 1;
        filter.rtrCode1 = 1;
        // No data bytes.
        filter.dataMask[0] = filter.dataMask[1] = 0xff;
        filter.dataCode[0] = filter.dataCode[1] = 0xff;
        //
        // Initialize board.
        //
        txCan = canInit( irqNumber , ioAddress , baudrate , &filter , 16 , 16 );
        //
        // If first board initialized successfully.
        if ( txCan != NULL )
        {
           // If second board.
           if ( ioAddress2 != 0 )
           {
              // Initialize second board.
              // Note that this board uses the same filter as the first board,
              // because all cantalk nodes will accept group 3 messages with
              // the prescribed first 5 bits.
              rxCan = canInit( irqNumber2 , ioAddress2 , baudrate , &filter , 16 , 16 );
           }
           else
              // Use the same board for receive and transmit.
              rxCan = txCan;
        }
        // If error initializing board.
        if ( ( txCan == NULL ) || ( rxCan == NULL ) )
        {
           // Show error in message area.
           ebErrorMsg.putStr( "Error initializing CAN board(s)" );
           ebErrorMsg.show();
           // Put cursor back.
           scrn1.showCursor();
        }
        else
           // Successful board init.
           canBoardInit = TRUE;
        // Endif
}
//
// ----------------------------------------------------------------------------
//
// Here to put a message out on to the CAN bus.
//
void sendMessage( char *s )
{
        char            c , *dst;
        int             done;
        canBuffer       buf;

        // Create header.
        buf.type = 0;
        buf.id = ( GROUP3_MSG_ID << 6 ) + rxMacId;
        buf.rtr = 0;
        // Put destination mac id as first data byte.
        buf.data[0] = txMacId;
        // Clear flag.
        done = FALSE;
        // Send until done.
        // First call to 'canSend()' will load the can chip with data; second
        // and subsequent calls will load the contents of the 'canBuffer' into
        // a ring buffer for transmission later (when the end of the previous
        // message generates a tx interrupt).
        while ( done == FALSE )
        {
           // Point to second byte in data buffer.
           dst = &buf.data[1];
           // If string + CR longer than 7 bytes.
           if ( strlen( s ) >= 7 )
           {
              // Sending 8 bytes.
              buf.cnt = 8;
              // Add bytes to data buffer.
              for ( c = 7 ; c-- ; ) *dst++ = *s++;
           }
           else
           {
              // We're done.
              done = TRUE;
              // Sending at least 2 byte.
              buf.cnt = 2;
              // Add bytes to data buffer.
              while ( *s )
              {
                 ++buf.cnt;
                 *dst++ = *s++;
              }
              // Add carriage return.
              *dst = CR;
           }
           // Now send the buffer.
           canSend( txCan , &buf );
        }
}
//
// ----------------------------------------------------------------------------
//
//      Special functions to customize operation of screen devices.
//
// ----------------------------------------------------------------------------
//
// Acceptance filters for edit boxes.
//
char	afHexDigit[] = "x0123456789abcdefABCDEF";
char	afDigit[] = "0123456789";
char    afBaudrate[] = "1234";
//
// ----------------------------------------------------------------------------
//
// Function to check for a valid PC I/O address.
//
int validIoAddress( char *s )
{
	int     ioVal;

	// Convert the string to integer.
	sscanf( s , "%x" , &ioVal );
	// Failure if any bits above A9 set.
	if ( ioVal & 0xffc00 )
	{
	   // Set error string.
	   scrnDev::errorPtr = "Invalid I/O Address";
	   // And return badly.
	   return FAILURE;
	}
	else
	   return SUCCESS;
}
//
// ----------------------------------------------------------------------------
//
// Function to check for valid IRQ number (supported by our cards).
//
int validIrqNumber( char *s )
{
	int     irqVal , retVal = SUCCESS;

	// Convert the string to integer.
	sscanf( s , "%i" , &irqVal );
	// Failure if incorrect irq number.
	switch ( irqVal )
	{
	case 0:
	case 2:
	case 9:
	case 3:
	case 4:
	case 5:
	case 6:
	case 7:
	case 10:
	case 11:
	case 12:
	case 14:
	case 15:
	   break;
	default:
	   // Set error string.
	   scrnDev::errorPtr = "Invalid IRQ Number";
	   // And return badly.
	   retVal = FAILURE;
	}
	// And return.
	return retVal;
}
//
// ----------------------------------------------------------------------------
//
// Function to check for a valid CAN Baudrate.
//
int validBaudrate( char *s )
{
	int     baud;

	// Convert the string to integer.
	sscanf( s , "%i" , &baud);
	// Failure if incorrect mac id.
	if ( ( baud < 1 ) || ( baud > 4 ) )
	{
	   // Set error string.
	   scrnDev::errorPtr = "Invalid Baudrate";
	   // And return badly.
	   return FAILURE;
	}
	else
	   return SUCCESS;
}
//
// ----------------------------------------------------------------------------
//
// Function to check for a valid MAC ID between 0 and 63.
//
int validMacId( char *s )
{
	int     macVal;

	// Convert the string to integer.
	sscanf( s , "%i" , &macVal );
	// Failure if incorrect mac id.
	if ( ( macVal < 0 ) || ( macVal >= 64 ) )
	{
	   // Set error string.
	   scrnDev::errorPtr = "Invalid MAC ID";
	   // And return badly.
	   return FAILURE;
	}
	else
	   return SUCCESS;
}
//
// ----------------------------------------------------------------------------
//
// Here to check for carriage return in 'txMessage', and to send the
// contents of the box out on to the CAN network.
// This function is used instead of deriving a new class from
// 'editBox'.
//
int txPreprocessFcn( int key )
{
        // Return with key value if not carriage return.
        if ( key != CR ) return key;

        #if     !LOOPBACK_MODE

        // These static strings are compared with the contents of the edit boxes
        // to determine if the operating data has changed.
        static char     sIoAddress[16],sIrqNumber[16],sBaudrate[16],sRxMacId[16],sTxMacId[16];

        // If board(s) initialized.
        if ( canBoardInit == TRUE )
        {
           // See if operating parameters have changed.
           if ( strcmp( sIoAddress , ebIoAddress.buf ) != 0 )
           {
              strcpy( sIoAddress , ebIoAddress.buf );
              canBoardInit = FALSE;
           }
           if ( strcmp( sIrqNumber , ebIrqNumber.buf ) != 0 )
           {
              strcpy( sIrqNumber , ebIrqNumber.buf );
              canBoardInit = FALSE;
           }
           if ( strcmp( sBaudrate , ebBaudrate.buf ) != 0 )
           {
              strcpy( sBaudrate , ebBaudrate.buf );
              canBoardInit = FALSE;
           }
           if ( strcmp( sRxMacId , ebRxMacId.buf ) != 0 )
           {
              strcpy( sRxMacId , ebRxMacId.buf );
              canBoardInit = FALSE;
           }
           if ( strcmp( sTxMacId , ebTxMacId.buf ) != 0 )
           {
              strcpy( sTxMacId , ebTxMacId.buf );
              canBoardInit = FALSE;
           }
        }
        // If can boards haven't been initialized.
        if ( canBoardInit == FALSE )
        {
           // Initialize can board(s).
           // This will display an error message if board(s) can't be init.
           initializeCanBoards();
        }
        // Don't bother sending message if init failed.
        if ( canBoardInit == TRUE )
        {
           // Send data in txMessage edit box.
           sendMessage( ebTxMessage.buf );
        }

        #else   // LOOPBACK_MODE

	char	*s;

	// This code just adds the contents of the tx message box to the
        // rx list box, character by character; it doesn't use the can boards.
        //
        // Point to tx message.
        s = ebTxMessage.buf;
        // Transfer to rx message list box.
        while ( *s ) lbRxMessage.processChar( *s++ );
        // Add carriage return.
        lbRxMessage.processChar( CR );

        #endif

        // Now kill the tx message.
        ebTxMessage.putStr( "" );
        ebTxMessage.cursorPos = 0;
        // And show the box.
        ebTxMessage.show();
        ebTxMessage.showCursor();
        // Return showing we've handled it.
        return SUCCESS;
}
//
// ----------------------------------------------------------------------------
//
//      Functions to read and save last saved screen data.
//
// ----------------------------------------------------------------------------
//
// File open/close error bitmasks.
//
#define		CANTALK_CFG_OPEN	0x0001
#define         CANTALK_BAD_CFG_DATA    0x0002
#define		CANTALK_CFG_WRITE       0x0004
//
// ----------------------------------------------------------------------------
//
// Here to read the configuration files.
//
static int readConfig( void )
{
        char	tbuf[128];
        char    tmpIo[32] , tmpIrq[32] , tmpBaud[32] , tmpTxMac[32] , tmpRxMac[32];
	int	paramCnt , readErrors = 0;
	FILE	*handle;

	// Open 'cantalk.cfg'.
	handle = fopen( "cantalk.cfg" , "r" );
        // If file exists.
        if ( handle != NULL )
	{
	   // Get first (and only) line.
	   if ( fgets( tbuf , 127 , handle ) != NULL )
	   {
	      // Extract data.
	      paramCnt = sscanf( tbuf , "%s %s %s %s %s" , &tmpIo , &tmpIrq , &tmpBaud , &tmpTxMac , &tmpRxMac );
	      // If too few parameters.
	      if ( paramCnt != 5 )
		 // Set error.
		 readErrors |= CANTALK_BAD_CFG_DATA;
	      // Else check values.
	      else
	      {
		 // See if valid I/O address.
		 if ( validIoAddress( tmpIo ) != SUCCESS ) readErrors |= CANTALK_BAD_CFG_DATA;
		 // See if valid irq number.
		 if ( validIrqNumber( tmpIrq ) != SUCCESS ) readErrors |= CANTALK_BAD_CFG_DATA;
		 // See if valid baudrate.
		 if ( validBaudrate( tmpBaud ) != SUCCESS ) readErrors |= CANTALK_BAD_CFG_DATA;
		 // See if valid tx MAC ID.
		 if ( validMacId( tmpTxMac ) != SUCCESS ) readErrors |= CANTALK_BAD_CFG_DATA;
		 // See if valid rx MAC ID.
		 if ( validMacId( tmpRxMac ) != SUCCESS ) readErrors |= CANTALK_BAD_CFG_DATA;
	      }
	      // Close file.
	      fclose( handle );
	   }
	   else
	   {
	      // Set error.
	      readErrors |= CANTALK_BAD_CFG_DATA;
	   }
	}
        // Else set flag
        else
           readErrors |= CANTALK_CFG_OPEN;
        // If no errors.
        if ( readErrors == 0 )
        {
           // Transfer data to global storage.
           ebIoAddress.putStr( tmpIo );
           ebIrqNumber.putStr( tmpIrq );
           ebBaudrate.putStr( tmpBaud );
           ebTxMacId.putStr( tmpTxMac );
           ebRxMacId.putStr( tmpRxMac );
        }
        // Return with error status.
        return readErrors;
}
//
// ----------------------------------------------------------------------------
//
// Here to save configuration data.
//
static int saveConfig( void )
{
	int		saveErrors = 0;
	FILE		*handle;

        // Open new configuration file; 'cantalk.cfg'
        handle = fopen( "cantalk.cfg" , "w" );
        // If you can open the new file.
        if ( handle != NULL )
        {
           // Write data to file.
	   fprintf( handle , "%s %s %s %s %s" , ebIoAddress.buf ,
						ebIrqNumber.buf ,
						ebBaudrate.buf ,
						ebTxMacId.buf ,
						ebRxMacId.buf );
           // Close file.
	   fflush( handle );
	   fclose( handle );
        }
        else
        {
           // Set error flag.
	   saveErrors |= CANTALK_CFG_WRITE;
	}
        // And return.
        return saveErrors;
}
//
// ----------------------------------------------------------------------------
//
//      Main routine support functions.
//
// ----------------------------------------------------------------------------
//
// Character input pre-processor.
// This looks for and processes ANSI escape sequences for the arrow keys
// sent by hyper terminal.
// It returns NULL for characters that are within an escape sequence.
// It returns the value of the control key when the last character of an
// escape sequence has been input.
//
// Have added the 'escTimer' to allow single ESC characters to be accepted
// separately. They are normally considered to be the first character of a
// multi-character ANSI escape sequence.
//
static int preProcess( int key )
{
	static char	escFlag;		// True when in escape sequence.
        static char	ansiBuf[32];		// ESC sequence buffer.
        static int	ansiIdx;		// 'ansiBuf' index.

	// If not within escape sequence.
	if ( escFlag == FALSE )
        {
           // Return with key if not ESC.
           if ( key != ESC ) return key;
           // Start the timer; if someone hits the ESC key all by itself,
           // we want to know that.
           tmrStart( escTimer , 3 );
           // Start of escape sequence, set flag.
           escFlag = TRUE;
           // Initialize buffer & store character.
           ansiIdx = 1;
           ansiBuf[0] = ESC;
           // And return with nothing.
           return 0;
        }
        //
        // If you get here, you're within an escape sequence, and must
        // process characters.
        //
        // Store character.
        ansiBuf[ ansiIdx++ ] = key;
        // Null terminate.
        ansiBuf[ ansiIdx ] = 0;
        // If char was alpha.
        if ( isalpha( key ) )
        {
           // End of ESC sequence reached.
           escFlag = FALSE;
           // Stop key timer.
           tmrStop( escTimer );
           // Convert to cursor key.
           switch ( key )
           {
           // Up arrow.
           case 'A':
              key = UP_ARROW;
              break;
           // Down arrow.
           case 'B':
              key = DOWN_ARROW;
              break;
           // Right arrow.
           case 'C':
              key = RIGHT_ARROW;
              break;
           // Left arrow.
           case 'D':
              key = LEFT_ARROW;
              break;
           default:
              key = 0;
           }
           // Return with key value.
           return key;
        }
        // If you get here, you're still processing an escape sequence.
        // Must return 0.
        return 0;
}
//
// ----------------------------------------------------------------------------
//
// Here to get keyboard character, if one waiting.
// Returns '0' if no keyboard input.
//
static int getKey( void )
{
	int		key;

        #if     REDIRECT

        // Try to get input.
        key = comRecv( scrnCom );
        // Return ESC if timer times out.
        if ( tmrFlag( escTimer ) ) return ESC;
        // Return if no keypress.
        if ( key == NULL_CHAR ) return 0;
        // Translate control characters to IBM style keypress.
        key = preProcess( key );

        #else

        // Return if keypress not waiting.
        if ( kbhit() == 0 ) return 0;
        // Get keypress.
        key = getch();
        // If function key.
        if ( key == 0x0000 )
        {
           // Get next character make function code.
           key = getch() | 0x100;
        }

        #endif

	// Return with key.
	return key;
}
//
// ----------------------------------------------------------------------------
//
// Control break handler; this code is executed if user hits CTRL-C during
// program execution; a sure sign he wants out...
//
int breakHandler( void )
{
        // Close any open files.
	fcloseall();
        // Release CAN board(s).
        if ( txCan == rxCan )
           canDestroy( txCan );
        else
        {
           canDestroy( rxCan );
           canDestroy( txCan );
        }
	// Set to white on black.
	setMode( F_WHITE , B_BLACK , A_OFF );
        // Clear the screen.
        clrScreen();
        // And abort execution; exit code of '0' will end program.
        exit( 0 );
        // NOTREACHED; Just to keep the compiler from bitching...
        return 0;
}
//
// ----------------------------------------------------------------------------
//
//      Main routine.
//
// ----------------------------------------------------------------------------
//
// Main routine to send messages between between CAN boards.
//
// Command line arguments are used to specify a second CAN board ('rxCan'):
//      cantalk <ioAddress2> <irqNumber2>
//
// If no command line arguments are present, 'rxCan' will be the same as 'txCan'.
//
// As is usually the case, 98% of this program is operator interface.
// Most of the operator interface is done by functions defined in 'screen.cpp'.
// These functions take care of adding characters to the edit boxes and
// list boxes shown on the screen. The first part of 'main()' just defines these
// screen elements and adds them to the 'scrn1' object. The 'scrn1' object takes
// care of passing keystrokes to the appropriate edit/list box and moving the
// cursor around.
//
// The list boxes perform automatic data verification when the TAB key is pressed
// and the focus is changed. Data verification is done by a function specific to
// each edit box (though the verification function can be shared, as is the case
// with the MAC ID's).
//
void main( int argc , char *argv[] )
{
	char		*src;
	int		key;
        int             xRuler1 , xRuler2 , xRuler3;
        int             yRuler1 , yRuler2 , yRuler3;
	editBox		*ebp;
	int		i , j;
	char		tbuf[128];
	FILE		*handle;
        canBuffer       cbuf;

	// Initialize control-break handler.
	ctrlbrk( breakHandler );

	// Initialize com port.
	#if	REDIRECT
	scrnCom = comInit( TRUE , COM1 , _8BIT , _19200 , SCRN_RX_SIZE , SCRN_TX_SIZE );
	#endif

	// Set to white on black.
	setMode( F_WHITE , B_BLACK , A_OFF );

        // If command line arguments.
        if ( argc > 1 )
        {
           // Use 'i' as error flag; assume a problem.
           i = FAILURE;
           // If the right number.
           if ( argc == 3 )
           {
              // If the first is a valid i/o address.
              if ( validIoAddress( argv[1] ) == SUCCESS )
              {
                 // If the second is a valid irq number.
                 if ( validIrqNumber( argv[2] ) == SUCCESS )
                 {
                    // We're good to go.
                    i = SUCCESS;
                    // Extract i/o address.
                    sscanf( argv[1] , "%x" , &ioAddress2 );
                    // Extract irq number.
                    sscanf( argv[2] , "%i" , &irqNumber2 );
                 }
              }
           }
           // If there was a problem.
           if ( i == FAILURE )
           {
              // Clear the screen.
              clrScreen();
              // Show error.
              showStrJ( "Invalid or wrong number of command line arguments" , 40 , 12 , J_CENTER );
              showStrJ( "Program exiting..." , 40 , 14 , J_CENTER );
              // And exit.
              exit( 1 );
           }
        }

        // Initialize timer.
        escTimer = tmrInit();

	// Clear screen and print header.
        clrScreen();
	showStrJ( "Contemporary Control Systems" , 40 , 0 , J_CENTER );
        showStrJ( "CAN Talk Utility" ,  40 , 1 , J_CENTER );
	showStrJ( revisionStr , 40 , 2 , J_CENTER );
        showStrJ( revisionDate , 40 , 3 , J_CENTER );

        // Set rulers for display.
        xRuler1 = 22;                   // Left hand data/text center point.
        xRuler2 = 37;                   // Right hand messages left side.

        //---------------------------------------------------------------------
        //      Start of screen device initialization.
        //---------------------------------------------------------------------

        // Create screen devices and print headers for them.
        //
        // CAN board I/O address.
        showStrJ( "I/O Address (HEX):" , xRuler1  , 6 , J_RIGHT );      // Text label.
        ebIoAddress.init( xRuler1 , 6 , 5 );                            // Define the edit box.
        ebIoAddress.filter = afHexDigit;                                // Set the input filter.
        ebIoAddress.dataCheckFcn = validIoAddress;                      // Set the data validation function.
        ebIoAddress.putStr( "0x100" );                                  // Loads default data in edit box.
        // CAN board IRQ number.
	showStrJ( "IRQ Number:" , xRuler1 , 8 , J_RIGHT );
	ebIrqNumber.init( xRuler1 , 8 , 5 );
	ebIrqNumber.filter = afDigit;
	ebIrqNumber.dataCheckFcn = validIrqNumber;
        ebIrqNumber.putStr( "10" );
        // CAN network baud rate.
        showStrJ( "Baudrate (1,2,3,4):" , xRuler1 , 10 , J_RIGHT );
	ebBaudrate.init( xRuler1 , 10 , 1 );
	ebBaudrate.filter = afBaudrate;
	ebBaudrate.dataCheckFcn = validBaudrate;
        ebBaudrate.putStr( "2" );
        // RX MAC ID.
	showStrJ( "RX MAC ID:" , xRuler1 , 12 , J_RIGHT );
        ebRxMacId.init( xRuler1 , 12 , 5 );
        ebRxMacId.filter = afDigit;
	ebRxMacId.dataCheckFcn = validMacId;
        ebRxMacId.putStr( "20" );
        // TX MAC ID.
	showStrJ( "TX MAC ID:" , xRuler1 , 14 , J_RIGHT );
        ebTxMacId.init( xRuler1 , 14 , 5 );
        ebTxMacId.filter = afDigit;
	ebTxMacId.dataCheckFcn = validMacId;
        ebTxMacId.putStr( "21" );
        // TX message box.
        showStrJ( "Transmitted Messages" , xRuler2 , 5 , J_LEFT );
 	ebTxMessage.init( xRuler2 , 6 , 40 );
        ebTxMessage.preprocessFcn = txPreprocessFcn;                    // This function gets keys
                                                                        // before 'processChar()'.
        // RX message box.
        showStrJ( "Received Messages" , xRuler2 , 7 , J_LEFT );
        lbRxMessage.init( xRuler2 , 8 , 40 , 10 , 10 );
        lbRxMessage.scroll = TRUE;                                      // Make sure data scrolls.
        // Error message box.
        showStrJ( "Error Messages" , xRuler2 , 18 , J_LEFT );
        ebErrorMsg.init( xRuler2 , 19 , 40 );

        // Add screen elements to control array.
	scrn1.addWidget( &ebIoAddress );
	scrn1.addWidget( &ebIrqNumber );
        scrn1.addWidget( &ebBaudrate );
	scrn1.addWidget( &ebRxMacId );
	scrn1.addWidget( &ebTxMacId );
	scrn1.addWidget( &ebTxMessage );

        //---------------------------------------------------------------------
        //      End of screen device initialization.
        //---------------------------------------------------------------------

        // Add editing instructions.
        showStrJ( "Move to next box :" , xRuler1 , 17 , J_RIGHT );
        showStrJ( " TAB" , xRuler1 , 17 , J_LEFT );
        showStrJ( "Transmit message :" , xRuler1 , 18 , J_RIGHT );
        showStrJ( " CR" , xRuler1 , 18 , J_LEFT );
        showStrJ( "Exit Program :" , xRuler1 , 19 , J_RIGHT );
        showStrJ( " ESC" , xRuler1 , 19 , J_LEFT );

        // Load data from input file.
        // This function will overwrite the default data loaded into the
        // edit boxes above if no errors are encountered in the cfg file.
        // Default data will be used if read errors were encountered.
	if ( readConfig() != SUCCESS )
        {
           // Indicate error reading file.
           ebErrorMsg.putStr( "Error reading cantalk.cfg" );
        }

        // Set to blue on white for general display areas.
	setMode( F_BLUE , B_WHITE , A_BOLD );

        // Show screen control elements.
        // This will show all devices added to 'scrn1'.
	scrn1.show();

        // Show received message area.
        // Must do this manually because it's not part of 'scrn1'.
        lbRxMessage.show();

        // Show error message area.
        // Must do this manually because it's not part of 'scrn1'.
        ebErrorMsg.show();

        // Put cursor back where screen would have it.
        scrn1.showCursor();

        //---------------------------------------------------------------------
        //      This is the START of the 'business end' of the program.
        //---------------------------------------------------------------------
        //
	// Loop to process characters.
        for (;;)
	{
           for (;;)
           {
              // If received message.
              if ( canRecv( rxCan , &cbuf ) == SUCCESS )
              {
                 // If meant for us: first data byte of message contains
                 // destination MAC ID.
                 if ( cbuf.data[0] == rxMacId )
                 {
                    // Don't count first data character (it was the MAC ID).
                    --cbuf.cnt;
                    // Point to second data byte of packet, it's the first
                    // 'real' character.
                    src = &cbuf.data[1];
                    // Print the characters in the rx message list box.
                    while ( cbuf.cnt-- ) lbRxMessage.processChar( *src++ );
                    // Put the cursor back.
                    scrn1.showCursor();
                 }
              }
              // See if keyboard input.
              key = getKey();
              // Break if key hit.
              if ( key != 0 ) break;
           }
           // Quit if ESC key hit.
           if ( key == ESC ) break;
	   // Send key to screen handler.
           // The screen 'processChar()' function passes the key to
           // the screen device which has focus, and it's 'processChar()'
           // function deals with the key.
           // When you hit a CR from within the tx message box, the pre-process
           // function for that box sends the message and clears the box.
	   switch ( scrn1.processChar( key ) )
	   {
	   case FAILURE:
	      // Display error message.
	      ebErrorMsg.putStr( scrnDev::errorPtr );
	      // And update screen.
	      ebErrorMsg.show();
	      break;
	   default:
	      // Clear error message.
	      ebErrorMsg.putStr( "" );
	      // And update screen.
	      ebErrorMsg.show();
	      break;
           }
           // And put cursor back.
           scrn1.showCursor();
        }
        //---------------------------------------------------------------------
        //      This is the END of the 'business end' of the program.
        //---------------------------------------------------------------------

	// Set to white on black.
	setMode( F_WHITE , B_BLACK , A_OFF );

        // Save current configuration.
        saveConfig();

	// Clear screen.
	clrScreen();

        // Wait for characters to be sent.
        #if     REDIRECT
        tmrStart( escTimer , 5 );
        while ( tmrFlag( escTimer ) == 0 );
        #endif

	// Release timer.
	tmrDestroy( escTimer );

        // Release CAN board(s).
        if ( txCan == rxCan )
           canDestroy( txCan );
        else
        {
           canDestroy( rxCan );
           canDestroy( txCan );
        }

	// Shut down the com port and re-initialize uart for polled mode operation.
        #if     REDIRECT
	comDestroy( scrnCom );
	scrnCom = comInit( FALSE , COM1 , _8BIT , _19200 , 1 , 1 );
	scrnCom->initPoll = TRUE;	// Set this so 'destroy' doesn't mess with hardware.
	comDestroy( scrnCom );
        #endif

	// And then exit.
	exit( 0 );
}

